/* hotkey.c -- deal with Fn+<KEY> presses on a Toshiba laptop under Linux
 *
 * Copyright (c) 1997-99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: hotkey.c,v $
 * Revision 2.2  2000/01/15 10:34:57  jab
 * changed Fn detection to new /proc interface
 * fixed a whole slew of bugs so it actually works
 *
 * Revision 2.1  1999/08/09 20:25:24  jab
 * use the SCI to get the settings rather than the CMOS
 * added support for new HCI based Fn detection
 *
 * Revision 2.0  1998/08/30 10:02:38  jab
 * added in support for speaker volume changes with Fn+F4
 * changed Japanese font to 14 point to make it more readable
 * various minor bug fixes and code tidy ups
 *
 * Revision 1.7  1998/08/22 09:23:11  jab
 * now check to see if the hotkey in the pid file is still running
 * various small tidy ups in the code
 *
 * Revision 1.6  1998/07/25 15:36:50  jab
 * added the text for the -h option and fixed new multibyte locale support
 *
 * Revision 1.5  1998/06/27 21:45:29  jab
 * added monochrome option which is automatically selected if unable to
 * allocate all the colours, interrupts are now disabled while probing the
 * status of the Fn key
 *
 * Revision 1.4  1998/06/13 22:48:35  jab
 * new Fn key polling technique, now down to 5 times a second
 *
 * Revision 1.3  1998/06/13 15:58:02  jab
 * fixes to Fn key polling
 *
 * Revision 1.2  1998/05/04 17:18:39  jab
 * added in X11 code so it now works the same as under Microsoft Windows
 *
 * Revision 1.1  1998/04/19 10:39:33  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static char const rcsid[]="$Id: hotkey.c,v 2.2 2000/01/15 10:34:57 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<asm/io.h>
#include<unistd.h>
#include<features.h>
#include<signal.h>
#include<sys/types.h>
#include<paths.h>

/* this deals with the differences between libc5 and glib2 */
#ifdef __GLIBC__
#include<sys/perm.h>
#else
#define X_LOCALE
#endif

#include<X11/Xlib.h>
#include<X11/Xlocale.h>


#include"hotkey.h"
#include"toshiba.h"
#include"sci.h"


static int popup = 0;
static int type = MAUI;
static int language = ENGLISH;
static int oldpower = -1;
static int oldbattery = -1;
static int oldvolume = -1;

static Display *display = NULL;
static Screen *scrptr;
static int screen;
static Window box;
static Colormap cmap;
static GC gc;
static XFontSet font;
static unsigned long pixels[14];
static int depth = 0;


#define FONT "-adobe-helvetica-bold-r-normal-*-12-*-*-*-*-*-*-*,\
-*-*-*-r-normal--14-*-*-*-*-*"

#define PID_FILE _PATH_VARRUN "hotkey.pid"

#define USAGE \
"Usage: hotkey [OPTION]\n\n\
Display a popup window on changing the battery save, power-up mode or\n\
speaker volume a Toshiba laptop using the Fn key combinations\n\n\
  -m    Display the popup window in monochrome rather than colour.\n\
  -de   Display text strings in German.\n\
  -en   Display text strings in English.\n\
  -es   Display text strings in Spanish.\n\
  -fr   Display text strings in French.\n\
  -it   Display text strings in Italian.\n\
  -pt   Display text strings in Portuguese.\n\
  -ja   Display text strings in Japanese.\n\n\
Report bugs to jonathan@buzzard.org.uk\n\
"

/*
 * Determine the model type for the different battery save settings
 */
int GetModelType(void)
{
	SMMRegisters reg;
	int model;

	SciOpenInterface();
	reg.ebx = SCI_BATTERY_SAVE;
	SciGet(&reg);
	SciCloseInterface();
	
	switch (reg.edx & 0xffff) {
		case 2:
			model = MAUI;
			break;
		case 3:
			model = STOCKHOLM;
			break;
		default:
			model = MAUI;
			break;
	}

	return model;
}


/*
 * Create a window 440 pixels wide by 80 pixels high in the centre of the screen
 */
void CreateBox(void)
{
	XSetWindowAttributes xattributes;
	int x,y;
	long mask;

	x = (DisplayWidth(display,0)-440)/2;
	y = (DisplayHeight(display, 0)-80)/2;

	box = XCreateSimpleWindow(display, RootWindow(display, screen),
	            x, y, 440, 80, 0, 0L, 0L);

	xattributes.save_under = True;
	xattributes.override_redirect = True;
	xattributes.cursor = None;
	mask = CWSaveUnder | CWOverrideRedirect;
	XChangeWindowAttributes(display, box, mask, &xattributes);

	mask = ExposureMask;
	XSelectInput (display, box, mask);

	XSetWindowBorderWidth(display, box, 0);

	/* set name so window manager can be set NOT to draw decorations */

	XStoreName(display, box, "hotkey");

	/* map window */

	XMapWindow(display, box);
	XFlush(display);

	return;
}


/*
 * Get rid of the popup window
 */
void DestroyBox(void)
{
	XUnmapWindow(display, box);
	XDestroyWindow(display, box);
	XFlush(display);
	oldpower = -1;
	oldbattery = -1;

	return;
}


/*
 * Paint the popup windows main elements with resume, power or volume icons
 */
void PaintBox(int style)
{
	XWindowAttributes attrib;
	int x,y,width,height;

	XGetWindowAttributes(display, box, &attrib);
	width = attrib.width;
	height = attrib.height;

	XSetForeground(display, gc, pixels[BLACK]);
	XDrawRectangle(display, box, gc, 0, 0, width-1, height-1);
	XSetForeground(display, gc, pixels[SILVER]);
	XFillRectangle(display, box, gc, 1, 1, width-2, height-2);
	XSetForeground(display, gc, pixels[GRAY]);
	XDrawLine(display, box, gc, 4, 4, width-5, 4);
	XDrawLine(display, box, gc, 4, 4, 4, height-5);
	XSetForeground(display, gc, pixels[WHITE]);
	XDrawLine(display, box, gc, 4, height-4, width-4, height-4);
	XDrawLine(display, box, gc, width-4, height-4, width-4, 4);

	if (style==0x3c) {
		for(y=0;y<32;y++)
			for(x=0;x<32;x++)
				if (fullpower[y][x]!=0x20) {
					XSetForeground(display, gc, pixels[fullpower[y][x]-0x30]);
					XDrawPoint(display, box, gc, 23+x, 25+y);
				}
		for(y=0;y<32;y++)
			for(x=0;x<32;x++)
				if (lowpower[y][x]!=0x20) {
					XSetForeground(display, gc, pixels[lowpower[y][x]-0x30]);
					XDrawPoint(display, box, gc, 84+x, 25+y);
				}
		for(y=0;y<32;y++)
			for(x=0;x<32;x++)
				if (userpower[y][x]!=0x20) {
					XSetForeground(display, gc, pixels[userpower[y][x]-0x30]);
					XDrawPoint(display, box, gc, 145+x, 25+y);
				}
	}

	if (style==0x3d) {
		for(y=0;y<32;y++)
			for(x=0;x<32;x++)
				if (resumeon[y][x]!=0x20) {
					XSetForeground(display, gc, pixels[resumeon[y][x]-0x30]);
					XDrawPoint(display, box, gc, 84+x, 25+y);
				}
		for(y=0;y<32;y++)
			for(x=0;x<32;x++)
				if (resumeoff[y][x]!=0x20) {
					XSetForeground(display, gc, pixels[resumeoff[y][x]-0x30]);
					XDrawPoint(display, box, gc, 145+x, 25+y);
				}
	}

	if (style==0x3e) {
		for(y=0;y<32;y++)
			for(x=0;x<32;x++)
				if (speakeroff[y][x]!=0x20) {
					XSetForeground(display, gc, pixels[speakeroff[y][x]-0x30]);
					XDrawPoint(display, box, gc, 23+x, 25+y);
				}
		for(y=0;y<32;y++)
			for(x=0;x<32;x++)
				if (lowvolume[y][x]!=0x20) {
					XSetForeground(display, gc, pixels[lowvolume[y][x]-0x30]);
					XDrawPoint(display, box, gc, 84+x, 25+y);
				}
		for(y=0;y<32;y++)
			for(x=0;x<32;x++)
				if (mediumvolume[y][x]!=0x20) {
					XSetForeground(display, gc, pixels[mediumvolume[y][x]-0x30]);
					XDrawPoint(display, box, gc, 145+x, 25+y);
				}
		for(y=0;y<32;y++)
			for(x=0;x<32;x++)
				if (highvolume[y][x]!=0x20) {
					XSetForeground(display, gc, pixels[highvolume[y][x]-0x30]);
					XDrawPoint(display, box, gc, 206+x, 25+y);
				}
	}

	XFlush(display);

	return;
}


/*
 * Draw a selection box around the correct icon and remove any existing boxes
 */
void SelectionBox(int draw)
{
	int x;

	/* erase any existing selection boxs */

	XSetForeground(display, gc, pixels[SILVER]);
	XDrawRectangle(display, box, gc, 15, 16, 47, 46);
	XDrawRectangle(display, box, gc, 76, 16, 47, 46);
	XDrawRectangle(display, box, gc, 137, 16, 47, 46);
	XDrawRectangle(display, box, gc, 198, 16, 47, 46);

	/* draw the selection box */

	if (draw<4) {
		x = 15+(61*draw);
		XSetForeground(display, gc, pixels[WHITE]);
		XDrawLine(display, box, gc, x, 16, x+46, 16);
		XDrawLine(display, box, gc, x, 16, x , 61);
		XSetForeground(display, gc, pixels[GRAY]);
		XDrawLine(display, box, gc, x+47, 16, x+47, 62);
		XDrawLine(display, box, gc, x, 62, x+47 , 62);
	}

	return;
}


void UpdateStatus(int key)
{
	SMMRegisters reg;
	int battery,power,volume,index;


	/* get battery save mode, power-up mode and volume setting */

	SciOpenInterface();
	reg.ebx = SCI_BATTERY_SAVE;
	SciGet(&reg);
	battery = (int) ((reg.edx & 0xffff)-(reg.ecx & 0xffff));
	reg.ebx = SCI_POWER_UP;
	SciGet(&reg);
	power = (int) (reg.ecx & 0xffff);
	reg.ebx = SCI_SPEAKER_VOLUME;
	SciGet(&reg);
	volume = (int) (reg.ecx & 0xffff);
	SciCloseInterface();

	if (battery<0)
		battery=2;
	if (power<0)
		power=1;

	/* exit if the mode has not changed */

	if ((battery==oldbattery) && (key==0xbc))
		return;
	if ((power==oldpower) && (key==0xbd))
		return;
	if ((volume==oldvolume) && (key==0xbe))
		return;

	/* display the status information */

	key &= 0x7f;
	if (key==0x3c) {
		switch (battery) {
			case 0:
				SelectionBox(0);
				index = 2;
				break;
			case 1:
				SelectionBox(1);
				index = (type==STOCKHOLM) ? 1 : 0;			
				break;
			case 2:
				SelectionBox(2);
				index = (type==STOCKHOLM) ? 0 : 3;
				break;
			case 3:
				SelectionBox(3);
				index = 3;
				break;
			default:
				/* this leaves no selection boxes */
				index = 0;
				SelectionBox(5); 
				break;
		}
		XSetForeground(display, gc, pixels[SILVER]);
		XFillRectangle(display, box, gc, 258, 32, 166, 15);
		XSetForeground(display, gc, pixels[BLACK]);
		XmbDrawString(display, box, font, gc, 259, 43,
			(save[language][index]),
				strlen(save[language][index]));
	}

	if (key==0x3d) {
		SelectionBox(power==0 ? 2 : 1);
		XSetForeground(display, gc, pixels[SILVER]);
		XFillRectangle(display, box, gc, 258, 32, 166, 15);
		XSetForeground(display, gc, pixels[BLACK]);
		XmbDrawString(display, box, font, gc, 259, 43,
			(boot[language][power]), strlen(boot[language][power]));
	}

	if (key==0x3e) {
		SelectionBox(volume);
		XSetForeground(display, gc, pixels[SILVER]);
		XFillRectangle(display, box, gc, 258, 32, 166, 15);
		XSetForeground(display, gc, pixels[BLACK]);
		XmbDrawString(display, box, font, gc, 259, 43,
			(speaker[language][volume]),
				strlen(speaker[language][volume]));
	}

	XFlush(display);

	/* update the saved battery and power modes */

	oldbattery = battery;
	oldpower = power;
	oldvolume = volume;

	return;
}


/*
 * catch any signals and exit the program in nicely.
 */
void catch(int x)
{
	fprintf(stderr, "hotkey: caught signal %d, cleaning up...\n", x);

	if (popup!=0)
		DestroyBox();

	if (depth>=8)
		XFreeColors(display, cmap, pixels, 10, 0);
	XFreeFontSet(display, font);
	XCloseDisplay(display);

	unlink(PID_FILE);

	exit(x);

	return;
}


int main(int argc, char *argv[])
{
	int count,key,loop,refresh,pid,version;
	long mask;
	FILE *str;
	XEvent event;
	XColor color;
	char **list,*def_string,*locale,buffer[64];


	/* do some quick checks on the laptop */

	if (SciSupportCheck(&version)==1) {
		fprintf(stderr, "hotkey: this computer is not supported\n");
		return 1;
	}

	/* check to see if a copy of hotkey is already running */

	if (!access(PID_FILE, R_OK)) {
		if ((str = fopen(PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other hotkey is still running */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "hotkey: Already running as "
					"process %d.\n", pid);
				return 1;
			}

			fprintf(stderr, "hotkey: process %d appears to have "
				"died, continuing\n", pid);
			unlink(PID_FILE);
		}
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}

	/* if the LANG enviroment variable sets a locale, get the language */

	locale = getenv("LANG");
	if (locale==NULL)
		locale = "C";
	else {
		if (!strncmp(locale, "en", 2))
			language = ENGLISH;
		if (!strncmp(locale, "de", 2))
			language = GERMAN;
		if (!strncmp(locale, "es", 2))
			language = SPANISH;
		if (!strncmp(locale, "fr", 2))
			language = FRENCH;
		if (!strncmp(locale, "it", 2))
			language = ITALIAN;
		if (!strncmp(locale, "pt", 2))
			language = PORTUGUESE;
		if (!strncmp(locale, "ja", 2))
			language = JAPANESE;
	}

	/* process the command line options, which overide the enviroment */

	if (--argc>0) {
		argv++;
		if (!strcmp(argv[0], "-h")) {
			printf(USAGE);
			unlink(PID_FILE);
			return 0;
		}		
		if (!strcmp(argv[0], "-m") || !strcmp(argv[0], "--mono")) {
			argv++;
			depth = 1;
			}
		if (!strcmp(argv[0], "-en")) {
			language = ENGLISH;
			locale = "C";
		}
		if (!strcmp(argv[0], "-de")) {
			language = GERMAN;
			locale = "de_DE.ISO8859-1";
		}
		if (!strcmp(argv[0], "-es")) {
			language = SPANISH;
			locale = "es_ES.ISO8859-1";
		}
		if (!strcmp(argv[0], "-fr")) {
			language = FRENCH;
			locale = "fr_FR.ISO8859-1";
		}
		if (!strcmp(argv[0], "-it")) {
			language = ITALIAN;
			locale = "it_IT.ISO8859-1";
		}
		if (!strcmp(argv[0], "-pt")) {
			language = PORTUGUESE;
			locale = "pt_PT.ISO8859-1";
		}
		if (!strcmp(argv[0], "-ja")) {
			language = JAPANESE;
			locale = "ja_JP.eucJP";
		}
	}

	/* try and set the locale */

	if (setlocale(LC_ALL, locale)==NULL) {
		fprintf(stderr, "hotkey: cannot set locale.\n");
		unlink(PID_FILE);
		return 1;
	}

	if (!XSupportsLocale()) {
		fprintf(stderr, "hotkey: X does not support locale \"%s\".\n",
			setlocale(LC_ALL, NULL));
		unlink(PID_FILE);
		return 1;
	}
 
	/* try and open the local display */

	if(!(display = XOpenDisplay(":0.0"))) {
		fprintf(stderr, "hotkey: unable to open display\n");
		unlink(PID_FILE);
		return 1;
	}

	scrptr = ScreenOfDisplay(display, screen);
	gc = DefaultGCOfScreen(scrptr);
	if (depth==0)
		depth = DefaultDepth(display, screen);

	/* try allocating colours if we have a colour screen */

	if (depth>=8) {
		cmap = DefaultColormapOfScreen(scrptr);

		/* allocate colors in the color map */

		color.flags = DoRed|DoGreen|DoBlue;
		for (loop=0;loop<10;loop++) {
			color.red = rgb[loop][0];
			color.green = rgb[loop][1];
			color.blue = rgb[loop][2];		
			if (!XAllocColor(display, cmap, &color)) {
				fprintf(stderr, "hotkey: XAllocColor failed on "
					"color %d, reverting to monochrome.\n",
					loop);
				XFreeColors(display, cmap, pixels, loop, 0);
				depth = 1;
				break;
			}
			pixels[loop] = color.pixel;
		}

		/* add in the repeat colours used in the monochrome handling */

		pixels[10] = pixels[RED];
		pixels[11] = pixels[GRAY];
		pixels[12] = pixels[WHITE];
		pixels[13] = pixels[SILVER];
	}		

	/* if we are using monochrome mode set the pallete accordingly */

	if (depth==1) {
		for (loop=0;loop<14;loop++) {
			if (mono[loop]==1)
				pixels[loop]=BlackPixelOfScreen(scrptr);
			else
				pixels[loop]=WhitePixelOfScreen(scrptr);
		}
	}

	/* load the font we are going to use */

	font = XCreateFontSet(display, FONT, &list, &count, &def_string);

	/* display warning if there are charsets for which fonts can be found */

	if (count>0) {
		fprintf(stderr, "hotkey: The following charsets are missing:\n");
		for(loop=0;loop<count;loop++)
			fprintf(stderr, "hotkey: \t%s\n", list[loop]);
        	XFreeStringList(list);
	}

	/* choose the events to handle */

	mask = ExposureMask;

	/* if we recieve a signal, exit nicely freeing up resources */

	signal(SIGHUP, catch);
	signal(SIGINT, catch);
	signal(SIGQUIT, catch);
	signal(SIGILL, catch);
	signal(SIGTRAP, catch);
	signal(SIGABRT, catch);
	signal(SIGIOT, catch);
	signal(SIGFPE, catch);
	signal(SIGKILL, catch);
	signal(SIGSEGV, catch);
	signal(SIGPIPE, catch);
 	signal(SIGTERM, catch);
	signal(SIGCHLD, catch);
	signal(SIGCONT, catch);
	signal(SIGSTOP, catch);
	signal(SIGTSTP, catch);
	signal(SIGTTIN, catch);
	signal(SIGTTOU, catch);

	popup = 0;
	refresh = 0;
	type = GetModelType();

	/* loop forever */

	for (;;) {

		/* sleep for 1/5th of a second */

		usleep(200000);

		/* open /proc/toshiba for reading */

		if (!(str = fopen(TOSH_PROC, "r")))
			return -1;

		/* scan in Fn key information */

		fgets(buffer, sizeof(buffer)-1, str);
		fclose(str);
		buffer[sizeof(buffer)-1] = '\0';
		sscanf(buffer, "%*s %*x %*d.%*d %*d.%*d %*x %x\n", &key);

		/* get rid of status window if Fn key released */

		if ((key==0x00) && (popup!=0)) {
			DestroyBox();
			popup = 0;
			continue;
		}

		/* do we have a Fn+F2/Fn+F3 key press to deal with? */

		if ((key==0x3c) || (key==0x3d) || (key==0x3e) ||
			(key==0xbc) || (key==0xbd) || (key==0xbe)) {

			if (popup==0) {
				CreateBox();
				popup = key & 0x7f;
				PaintBox(popup);
			}
			if ((key & 0x7f)==popup) {
				UpdateStatus(key);
			}

			/* event loop handler */

			if (XCheckMaskEvent(display, mask, &event)==True) {
				switch (event.type) {
					case Expose:
						if (event.xexpose.count!=0)
							break;
						refresh = 1;
						break;
					case ExposureMask:
					case VisibilityNotify:
						refresh = 1;
					default:
						break;
				}
			}

			if (refresh==1) {
				PaintBox(popup);
				UpdateStatus(popup);
				refresh = 0;
			}
		}
	}

	return 0;
}
