/* tpasswd.c -- enter/delete the user password on a Toshiba laptop.
 *
 * Copyright (c) 1998/99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: tpasswd.c,v $
 * Revision 1.2  1999/08/09 20:22:38  jab
 * changed the usage string from the tpasswd one.
 *
 * Revision 1.1  1999/07/25 14:53:06  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

static const char rcsid[]="";

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif
#include<signal.h>
#include<termios.h>
#include<fcntl.h>
#include<paths.h>
#include<sys/ioctl.h>
#include<sys/vt.h>
#include<sys/kd.h>

#include"sci.h"

static int interface = 0;     /* is there an open interface to the laptop */
static int keyboard = -1;
static int terminal = 0;
static int fd;
struct termio saved;


#define PID_FILE _PATH_VARRUN "tpasswd.pid"

#define USAGE \
"Register/remove the user password and change the\n\n\
Usage: tpasswd [-h]\n\
    without parameters : Display if the user password is registered or not,\n\
                         and allow it to be registered/removed.\n\
    -h : Display this message.\n"



/*
 * Exit the program closing any open interface to the BIOS if necessary
 */
void leave(int close)
{
	if (interface==1)
		SciCloseInterface();
	unlink(PID_FILE);

	exit(close);
	return;
}


/*
 * Catch any signals and exit the program in nicely.
 */
void catch(int x)
{
	fprintf(stderr, "tpasswd: caught signal %d, cleaning up...\n", x);
	if (keyboard!=-1)
		ioctl(fd, KDSKBMODE, keyboard);

	if (terminal==1)
		close(fd);

	leave(1);
	

	return;
}


/*
 * Print Incompatiable machine type and exit
 */
void Incompatible()
{
	fprintf(stderr, "tpasswd: Incompatiable machine type.\n");
	leave(1);
}


/*
 * Put terminal into raw mode and then get a single keypress
 */
int GetScanCode()
{
	struct termio new;
	unsigned char key[1];

	/* exit if unable to open the console */

  	if ((fd = open("/dev/tty", O_RDWR))<0) {
		fprintf(stderr, "tpasswd: unable to open console\n");
		leave(1);
	}
	terminal = 1;

	/* get the current terminal state */

	ioctl(fd, KDGKBMODE, &keyboard);
	ioctl(fd, TCGETA, &saved);
	ioctl(fd, TCGETA, &new);

	/* set appropriate terminal mode */

	new.c_lflag &= ~ (ICANON | ECHO | ISIG); 
	new.c_iflag = 0;
	new.c_cc[VMIN] = 1;
	new.c_cc[VTIME] = 1;

	ioctl(fd, TCSETAW, &new);
	ioctl(fd, KDSKBMODE, K_RAW);

	key[0] = 0;
	read(fd, key, 1);

	ioctl(fd, TCSETAW, &saved);
	ioctl(fd, KDSKBMODE, keyboard);
	keyboard = -1;

	/* close the connection to the terminal  */

	close(fd);
	terminal = 0;

	return (int) key[0];
}


/*
 * Get a yes/no reply from the user as a keypress
 */
int YesNo(void)
{
	int reply;
	struct termio new;

	/* exit if unable to open the console */

	if ((fd = open("/dev/tty", O_RDWR))<0) {
		fprintf(stderr, "tpasswd: unable to open console\n");
		leave(1);
	}
	terminal = 1;

	/* get the current terminal state */

	ioctl(fd, TCGETA, &saved);
	ioctl(fd, TCGETA, &new);

	/* set appropriate terminal mode */

	new.c_iflag = (IXON|IXOFF);
	new.c_oflag = 0;
	new.c_lflag = 0;
	new.c_cc[VMIN] = 1;
	new.c_cc[VTIME] = 0;

	ioctl(fd, TCSETAW, &new);

	for(reply=-1;reply==-1;) {
		switch(getchar()) {
			case 'y': case'Y':
				reply = 1;
				break;
			case 'n': case 'N':
				reply = 0;
				break;
			default:
				printf("\a");
				break;
		}
	}

	/* restore the terminal mode and start a newline*/

	ioctl(fd, TCSETAW, &saved);
	close(fd);
	terminal = 0;
	printf("\n");

	return (reply);
}


/*
 * Get a password from the user echoing astrix's
 */
void GetPassword(char *password)
{
	int loop,scan;

	/* blank the password */

	for(loop=0;loop<10;loop++)
		*(password+loop) = 0;

	fflush(stdout);

	/* now get the password from the user*/

	for (loop=0;;) {
		scan = GetScanCode();
		if ((scan<0x02) || (scan>0x39)) continue;
		if ((scan==0x1d) || (scan==0x2a) || (scan==0x36)
		      || (scan==0x37) || (scan==0x38)) continue;
		if (scan==0x0f) continue;
		if (scan==0x1c) {
			printf("\n");
			fflush(stdout);
			break;
		}
		if (scan==0x0e) {
			if (loop>0) {
				printf("\b \b");
				fflush(stdout);
				loop--;
				*(password+loop) = 0;
			}
			continue;
		}
		if (loop<10) {
			*(password+loop) = scan;
			printf("*");
			fflush(stdout);
			loop++;
		}
	}

	return;
}


/*
 * Set/delete the User password
 */
void UserPassword(int registered, int trys)
{
	int reply;
	char enter[10],verify[10];

	if (registered==0) {
		printf("USER PASSWORD = Not Registered\n");
		printf("Do you want to register the user password <Y/N>?");
		reply = YesNo();
		while (reply==1) {
			printf("Enter password ---> ");
			GetPassword(enter);
			printf("Verify password --> ");
			GetPassword(verify);
			if (memcmp(enter, verify, 10)!=0) {
				printf("Password verify error !\nDo you want to"
					" retry <Y/N>?");
				reply = YesNo();
			} else {
				reply = 0;
				SciSetPassword(enter, SCI_USER_PASSWORD, &trys);
				printf("USER PASSWORD = Registered\n");
			}
		}
	} else {
		printf("USER PASSWORD = Registered\n");
		printf("Do you want to delete the user password <Y/N>?");
		reply = YesNo();
		while (reply==1) {
			if (trys==0) {
				printf("Password access denied !\n");
				leave(3);
			}
			printf("Enter password ---> ");
			GetPassword(enter);
			reply = 0;
			if (SciSetPassword(enter, SCI_USER_PASSWORD, &trys)!=SCI_SUCCESS) {
				printf("Password verify error !\nDo you want to"
					" retry <Y/N>?");
				reply = YesNo();
			} else {
				printf("USER PASSWORD = Not Registered\n");
				reply = 0;
			}
		}
	}

	return;
}


int main(int argc, char *argv[])
{
	SMMRegisters reg;
	int pid,help,version;
	FILE *str;
	struct vt_mode vtm;


	/* this program *must* be run as root */

	if (getuid()) {
		fprintf(stderr, "tpasswd: must be run as root.\n");
		return 1;
	}

	/* check that we are running at the console */

  	if ((fd = open("/dev/tty", O_RDWR))<0) {
		fprintf(stderr, "tpasswd: unable to open console.\n");
		return 1;
	}
	if (ioctl(fd, VT_GETMODE, &vtm)!=0) {
		fprintf(stderr, "tpasswd: not running at the console.\n");
		close(fd);
		return 1;
	}
	close(fd);

	/* process command line arguments */

 	help = 0;
	if ((--argc>0) && ((*++argv)[0]=='-'))
		switch (*++argv[0]) {
			case 'h':
				help = 1;
				break;
			default:
				printf("tpasswd: illegal option %s\n", argv[0]);
				return 1;
		}

	/* if we recieve a signal, exit cleanly closing any interface to the
	   BIOS, deleting run lock and putting the keyboard into usable state */

	signal(SIGHUP, catch);
	signal(SIGINT, catch);
	signal(SIGQUIT, catch);
	signal(SIGILL, catch);
	signal(SIGTRAP, catch);
	signal(SIGABRT, catch);
	signal(SIGIOT, catch);
	signal(SIGFPE, catch);
	signal(SIGKILL, catch);
	signal(SIGSEGV, catch);
	signal(SIGPIPE, catch);
 	signal(SIGTERM, catch);
	signal(SIGCHLD, catch);
	signal(SIGCONT, catch);
	signal(SIGSTOP, catch);
	signal(SIGTSTP, catch);
	signal(SIGTTIN, catch);
	signal(SIGTTOU, catch);

	/* check to see if a copy of tpasswd is already running */

	if (!access(PID_FILE, R_OK)) {
		if ((str = fopen(PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);
			fprintf(stderr, "tpasswd: Already running as process "
				"%d.\n", pid);
			return 1;
		}
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}
	
	/* is this a supported machine? */

	if (SciSupportCheck(&version)==SCI_FAILURE)
		Incompatible();

	/* did the user request help? */

	if (help==1) {
		printf(USAGE);
		leave(0);
	}

	/* Try opening an interface to the BIOS */

	if (SciOpenInterface()==SCI_SUCCESS)
		interface = 1;
	else
		Incompatible();

	/* is the user password registered? */

	reg.ebx = SCI_PASSWORD;
	reg.ecx = SCI_USER_PASSWORD;
	if (SciGet(&reg)==SCI_FAILURE)
		Incompatible();

	UserPassword((int) reg.ecx, (int) reg.edx);

	leave(0);

	return 0;
}
