/* tosh.c -- Linux driver for accessing the SMM on Toshiba laptops 
 *
 * Copyright (c) 1996-99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING
 *
 *   This code is covered by the GNU GPL and you are free to make any
 *   changes you wish to it under the terms of the license. However the
 *   code has the potential to render your computer and/or someone else's
 *   unuseable. Unless you truely understand what is going on, I urge you
 *   not to make any modifications and use it as it stands.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * The information used to write this driver has been obtained by reverse
 * engineering the software supplied by Toshiba for their portable computers in
 * strict accordance with the European Council Directive 92/250/EEC on the legal
 * protection of computer programs, and it's implementation into English Law by
 * the Copyright (Computer Programs) Regulations 1992 (S.I. 1992 No.3233).
 *
 * Compile it with:
 *
 *      gcc -O -DMODULE -D__KERNEL__ -c tosh.c
 *      gcc -O -c tosh_smm.s
 *      ld -r -S -o toshiba.o tosh.o tosh_smm.o
 *
 */

static char *version = "toshiba.c:v1.0 31/12/1999 Jonathan Buzzard (jonathan@buzzard.org.uk)\n";


#include<linux/kernel.h>
#include<linux/sched.h>
#include<linux/types.h>
#include<linux/fcntl.h>
#include<linux/miscdevice.h>
#include<linux/ioport.h>
#include<asm/io.h>

#ifdef MODULE
#include<linux/module.h>
#include<linux/version.h>
#endif
#ifdef CONFIG_PROC_FS
#include<linux/stat.h>
#include<linux/proc_fs.h>
#endif

#include"toshiba.h"

#define TOSH_MINOR_DEV 160

static int tosh_id = 0x0000;
static int tosh_bios = 0x0000;
static int tosh_date = 0x0000;
static int tosh_sci = 0x0000;
static int tosh_fn = 0;
static int tosh_fan = 0;


extern int tosh_smm(unsigned long *reg);
static int tosh_get_info(char *, char **, off_t, int, int);
static int tosh_open(struct inode *ip, struct file *);
static int tosh_release(struct inode *, struct file *);
static int tosh_ioctl(struct inode *, struct file *, unsigned int,
	unsigned long);


static struct file_operations tosh_fops = {
	NULL,		/* lseek */
	NULL,		/* read */
	NULL,		/* write */
	NULL,		/* readdir */  
	NULL,		/* select */ 
	tosh_ioctl,	/* ioctl */
	NULL,		/* mmap */
	tosh_open,	/* open */
	NULL,		/* flush */
	tosh_release, 	/* close */
	NULL,		/* fsync */
	NULL		/* fasync */
};

static struct miscdevice tosh_device = {
	TOSH_MINOR_DEV,
	"toshiba",
	&tosh_fops
};



/*
 * Read the Fn key status using the older none HCI method
 */
static int tosh_fn_status(void)
{
        unsigned char al;

	asm ("pushl %%eax\n\t" \
                "movw $0x008e,%%ax\n\t" \
		"cli\n\t" \
                "outb %%al,$0xe4\n\t" \
                "inb $0xe5,%%al\n\t" \
                "sti\n\t" \
                "movb %%al,%0\n\t" \
		"popl %%eax\n" \
                :"=m" (al) : : "memory" );

        return (int) al;
}


/*
 * Emulate the status of the Fn key
 *
 *   For models which don't support the HCI System Event method of detecting
 *   Fn key presses emulate them here.
 */
static int tosh_emulate_fn(unsigned long *regs)
{
	static int system;
	static int hotkey;
	int event;
	unsigned long eax,ebx,ecx;

	eax = regs[0] & 0xff00;
	ebx = regs[1] & 0x00ff;
	ecx = regs[2] & 0xffff;


	/* System event emulation */

	if ((ebx==0x0016) && (eax==0xfe00)) {
		if ((hotkey==1) && (system==1)) {
			event = tosh_fn_status();
			if (event==0) {
				regs[0] = 0x8c00;
				return 1;
			} else {
				regs[0] = 0x0044;
				regs[2] = event & 0x0100;
				return 0;
			}
		}
	}
	if ((ebx==0x0016) && (eax==0xff00)) {
		switch (ecx) {
			case 0x0000:
				system = 0;
				regs[0] = 0x0044;
				break;
			case 0x0001:
				system = 1;
				regs[0] = 0x0044;
				break;
			default:
				regs[0] = 0x8000;
				return 1;					
		}
		return 0;
	}

	/* Hotkey event emulation */

	if ((ebx==0x001e) && (eax==0xfe00)) {
		regs[0] = 0x0044;
		regs[2] = hotkey;
		/* to enable us to distinguish between the old and new methods
		   until the problem causing the lockups is solved return an
		   error here. */
		regs[0] = 0x8000;
		return 1;					
	}
	if ((ebx==0x001e) && (eax==0xff00)) {
		switch (ecx) {
			case 0x0000:
				hotkey = 0;
				regs[0] = 0x0044;
				break;
			case 0x0001:
				hotkey = 1;
				regs[0] = 0x0044;
				break;
			default:
				regs[0] = 0x8300;
				return 1;					
		}
	}

	return 0;
}

/*
 * For the Portage 610CT and the Tecra 700CS/700CDT emulate the HCI fan function
 */
static int tosh_emulate_fan(unsigned long *regs)
{
	unsigned long eax,ecx;
	unsigned short ax;

	eax = regs[0] & 0xff00;
	ecx = regs[2] & 0xffff;

	/* Portage 610CT */

	if (tosh_id==0xfccb) {
		if (eax==0xfe00) {
			/* fan status */
			asm ("pushl %%eax\n\t" \
				"pushl %%ebx\n\t" \
				"pushl %%ecx\n\t" \
				"cli\n\t" \
				"movb $0xbe,%%al\n\t" \
				"outb %%al,$0xe4\n\t" \
				"inb $0xe5,%%al\n\t" \
				"andw $0x0001,%%ax\n\t" \
				"movw %%ax,%0\n\t" \
				"sti\n\t" \
				"popl %%ecx\n\t" \
				"popl %%ebx\n\t" \
				"popl %%eax\n" \
 				:"=m" (ax) : : "memory" );
			regs[0] = 0x00;
			regs[2] = (ax==0x00) ? 0x00:0x01;
			return 0;
		}
		if ((eax==0xff00) && (ecx==0x0000)) {
			/* fan off */
			asm ("pushl %eax\n\t" \
				"pushl %ebx\n\t" \
				"pushl %ecx\n\t" \
				"cli\n\t" \
				"movb $0xbe,%al\n\t" \
				"outb %al,$0xe4\n\t" \
				"inb $0xe5,%al\n\t" \
				"orb $0x01,%al\n\t" \
				"movb %al,%ah\n\t" \
				"movb $0xbe,%al\n\t" \
				"outb %al,$0xe4\n\t" \
				"movb %ah,%al\n\t" \
				"outb %al,$0xe5\n\t" \
				"sti\n\t" \
				"popl %ecx\n\t" \
				"popl %ebx\n\t" \
				"popl %eax\n");
			regs[0] = 0x00;
			regs[2] = 0x00;
			return 0;
		}
		if ((eax==0xff00) && (ecx==0x0001)) {
			/* fan on */
			asm ("pushl %eax\n\t" \
				"pushl %ebx\n\t" \
				"pushl %ecx\n\t" \
				"cli\n\t" \
				"movb $0xbe,%al\n\t" \
				"outb %al,$0xe4\n\t" \
				"inb $0xe5,%al\n\t" \
				"andb $0xfe,%al\n\t" \
				"movb %al,%ah\n\t" \
				"movb $0xbe,%al\n\t" \
				"outb %al,$0xe4\n\t" \
				"movb %ah,%al\n\t" \
				"outb %al,$0xe5\n\t" \
				"sti\n\t" \
				"popl %ecx\n\t" \
				"popl %ebx\n\t" \
				"popl %eax\n");
			regs[0] = 0x00;
			regs[2] = 0x01;
			return 0;
		}
	}

	/* Tecra 700CS/CDT */

	if (tosh_id==0xfccc) {
		if (eax==0xfe00) {
			/* fan status */
			asm ("pushl %%eax\n\t" \
				"pushl %%ebx\n\t" \
				"pushl %%ecx\n\t" \
				"cli\n\t" \
				"movw $0x00e4,%%dx\n\t" \
				"movb $0xe0,%%al\n\t" \
				"outb %%al,%%dx\n\t" \
				"incw %%dx\n\t" \
				"inb %%dx,%%al\n\t" \
				"andw $0x0001,%%ax\n\t"
				"movw %%ax,%0\n\t" \
				"sti\n\t" \
				"popl %%ecx\n\t" \
				"popl %%ebx\n\t" \
				"popl %%eax\n" \
				:"=m" (ax) : : "memory" );
			regs[0] = 0x00;
			regs[2] = (unsigned long) ax;
		}
		if ((eax==0xff00) && (ecx==0x0000)) {
			/* fan off */
			asm ("pushl %eax\n\t" \
				"pushl %ebx\n\t" \
				"pushl %ecx\n\t" \
				"cli\n\t" \
				"movw $0x00e4,%dx\n\t" \
				"movb $0xe0,%al\n\t" \
				"outb %al,%dx\n\t" \
				"incw %dx\n\t" \
				"inb %dx,%al\n\t" \
				"andw $0xfffe,%ax\n\t" \
				"decw %dx\n\t" \
				"movb %al,%ah\n\t" \
				"movb $0xe0,%al\n\t" \
				"outw %ax,%dx\n\t" \
				"sti\n\t" \
				"popl %ecx\n\t" \
				"popl %ebx\n\t" \
				"popl %eax\n");
			regs[0] = 0x00;
			regs[2] = 0x00;
			return 0;
		}
		if ((eax==0xff00) && (ecx==0x0001)) {
			/* fan on */
			asm ("pushl %eax\n\t" \
				"pushl %ebx\n\t" \
				"pushl %ecx\n\t" \
				"cli\n\t" \
				"movw $0x00e4,%dx\n\t" \
				"movb $0xe0,%al\n\t" \
				"outb %al,%dx\n\t" \
				"incw %dx\n\t" \
				"inb %dx,%al\n\t" \
				"orw $0x0001,%ax\n\t" \
				"decw %dx\n\t" \
				"movb %al,%ah\n\t" \
				"movb $0xe0,%al\n\t" \
				"outw %ax,%dx\n\t" \
				"sti\n\t" \
				"popl %ecx\n\t" \
				"popl %ebx\n\t" \
				"popl %eax\n");
			regs[0] = 0x00;
			regs[2] = 0x01;
			return 0;
		}
	}

	return 0;
}


static int tosh_open(struct inode *ip, struct file *fp)
{
	MOD_INC_USE_COUNT;

	return 0;
}


static int tosh_release(struct inode *ip, struct file *fp)
{
	MOD_DEC_USE_COUNT;

	return 0;
}


static int tosh_ioctl(struct inode *ip, struct file *fp, unsigned int cmd,
	unsigned long arg)
{
	unsigned long *regs;
	unsigned short ax,bx;
	int err;

	if (!arg)
		return -EINVAL;

	regs = (unsigned long *) arg;
	switch (cmd) {
		case TOSH_SMM:
			ax = regs[0] & 0xff00;
			bx = regs[1] & 0xffff;
			/* block HCI calls to read/write memory & PCI devices */
			if (((ax==0xff00) || (ax==0xfe00)) && (bx>0x0069))
				return -EINVAL;

			/* do we need to emulate things for older models? */
			if (tosh_fan==1) {
				if (((ax==0xf3) || (ax==0xf4)) && (bx==0x0004)) {
					err = tosh_emulate_fan(regs);
					break;
				}
			}
			if (tosh_fn==1) {
				if (((ax==0xf3) || (ax==0xf4)) &&
					((bx==0x0016) || (bx==0x001e))) {
					err = tosh_emulate_fn(regs);
					break;
				}
			}
			err = tosh_smm(regs);
/*			printk("toshiba smm call exit:\n");
			printk("eax=0x%08x\nebx=0x%08x\necx=0x%08x\n",
				regs[0], regs[1], regs[2]);*/
			break;
		default:
			return -EINVAL;
	}

	return (err==0) ? 0:-EINVAL;
}


/*
 * Print the information for /proc/toshiba
 */
#ifdef CONFIG_PROC_FS
int tosh_get_info(char *buffer, char **start, off_t fpos, int length, int dummy)
{
	char *temp;

	temp = buffer;

	/* Arguments
	     0) Linux driver version (this will change if format changes)
	     1) Machine ID
	     2) SCI version
	     3) BIOS version (major, minor)
	     4) BIOS date (in SCI date format)
	*/

	temp += sprintf(temp, "1.0 0x%04x %d.%d %d.%d 0x%04x\n",
		tosh_id,
		(tosh_sci & 0xff00)>>8,
		tosh_sci & 0xff,
		(tosh_bios & 0xff00)>>8,
		tosh_bios & 0xff,
		tosh_date);

	return temp-buffer;
}
#endif


/*
 * Probe for the presence of a Toshiba laptop
 *
 *   returns and non-zero if unable to detect the presence of a Toshiba
 *   laptop, otherwise zero and determines the Machine ID, BIOS version and
 *   date, and SCI version.
 */
int tosh_probe(void)
{
	int major,minor,day,year,month;
	unsigned long address;
	unsigned short bx,cx,dx;
	unsigned char ah,flags;

	/* call the Toshiba SCI support check routine */

	asm ("pushl %%eax\n\t" \
		"pushl %%ebx\n\t" \
		"pushl %%ecx\n\t" \
		"pushl %%edx\n\t" \
		"movw $0xf0f0,%%ax\n\t" \
		"movw $0x0000,%%bx\n\t" \
		"movw $0x0000,%%cx\n\t" \
		"inb $0xb2,%%al\n\t" \
		"movb %%ah,%0\n\t" \
		"lahf\n\t" \
		"movb %%ah,%1\n\t" \
		"movw %%dx,%2\n\t" \
		"popl %%edx\n\t" \
		"popl %%ecx\n\t" \
		"popl %%ebx\n\t" \
		"popl %%eax\n" \
		:"=m" (ah), "=m" (flags), "=m" (dx) : : "memory" );

	/* if this is not a Toshiba laptop carry flag is set and ah=0x86 */

	if ((ah==0x86) || ((flags & 0x01)==1)) {
		printk("toshiba: not a supported Toshiba laptop\n");
		return -ENODEV;
	}

	/* if we get this far then we are running on a Toshiba (probably)! */

	tosh_sci = (int) dx;
	printk("toshiba: whee running on a Toshiba!!!\n");

	/* next get the machine ID of the current laptop */

	tosh_id = (0x100*(int) readb(0xffffe))+((int) readb(0xffffa));
	
	/* do we have a SCTTable machine identication number on our hands */

	if (tosh_id==0xfc2f) {

		/* start by getting a pointer into the BIOS */

		asm ("pushl %%eax\n\t" \
			"pushl %%ebx\n\t" \
			"pushl %%ecx\n\t" \
			"pushl %%edx\n\t" \
			"movw $0xc000,%%ax\n\t" \
			"movw $0x0000,%%bx\n\t" \
			"movw $0x0000,%%cx\n\t" \
			"inb $0xb2,%%al\n\t" \
			"movw %%bx,%0\n\t" \
			"movb %%ah,%1\n\t" \
			"popl %%edx\n\t" \
			"popl %%ecx\n\t" \
			"popl %%ebx\n\t" \
			"popl %%eax\n" \
			:"=m" (bx), "=m" (ah) \
			: \
			: "memory" );

		/* At this point in the Toshiba routines under MS Windows
		   the bx register holds 0xe6f5. However my code is producing
		   a different value! For the time being I will just fudge the
		   value. This has been verified on a Satellite Pro 430CDT,
		   Tecra 750CDT, Tecra 780DVD and Satellite 310CDT. */

/*		printk("toshiba: debugging ID bx=0x%04x\n", bx);*/
		bx = 0xe6f5;

		/* now twiddle with our pointer a bit */

		address = 0x000f0000+bx;
		cx = readw(address);
		address = 0x000f0009+bx+cx;
		cx = readw(address);
		address = 0x000f000a+cx;
		cx = readw(address);

		/* now construct our machine identification number */

		tosh_id = ((cx & 0xff)<<8)+((cx & 0xff00)>>8);
	}

	/* get the BIOS version */

	major = readb(0xfe009)-'0';
	minor = ((readb(0xfe00b)-'0')*10)+(readb(0xfe00c)-'0');
	tosh_bios = (major*0x100)+minor;

	/* get the BIOS date */

	day = ((readb(0xffff5)-'0')*10)+(readb(0xffff6)-'0');
	month = ((readb(0xffff8)-'0')*10)+(readb(0xffff9)-'0');
	year = ((readb(0xffffb)-'0')*10)+(readb(0xffffc)-'0');
	tosh_date = (((year-90) & 0x1f)<<10) | ((month & 0xf)<<6)
		| ((day & 0x1f)<<1);

	/* are we going to have to grap ports 0xe4 and 0xe5 for this model? */

	asm ("pushl %%eax\n\t" \
		"pushl %%ebx\n\t" \
		"pushl %%ecx\n\t" \
		"pushl %%edx\n\t" \
		"movw $0xfe00,%%ax\n\t" \
		"movw $0x001e,%%bx\n\t" \
		"movw $0x0000,%%cx\n\t" \
		"inb $0xb2,%%al\n\t" \
		"movb %%ah,%0\n\t" \
		"lahf\n\t" \
		"movb %%ah,%1\n\t" \
		"popl %%edx\n\t" \
		"popl %%ecx\n\t" \
		"popl %%ebx\n\t" \
		"popl %%eax\n" \
		:"=m" (ah), "=m" (flags) : : "memory" );

	if ((ah!=0x00) || ((flags & 0x01)==1)) {
		if (check_region(0xe4, 2)<0) {
			printk("toshiba: ERROR ioports 0xe4-0xe5 are already "
				"in use\n");
			return -ENODEV;
		}
		request_region(0xe4, 2, "toshiba");
		tosh_fn = 1;
        }

	/* do we need to emulate the fan? (they also need Fn emulation) */

	if ((tosh_id==0xfccb) || (tosh_id==0xfccc))
		tosh_fan = 1;

	return 0;
}


#ifdef MODULE
int init_module(void)
{
	static struct proc_dir_entry *ent;

	printk("%s", version);

	/* are we running on a Toshiba laptop */

	if (tosh_probe()!=0)
		return -EIO;

	/* register the proc entry */

	ent = create_proc_entry("toshiba", 0, 0);
	if (ent!=NULL)
		ent->get_info = tosh_get_info;


	/* register the device file */

	misc_register(&tosh_device);

	return 0;
}

void cleanup_module(void)
{
	/* remove the proc entry */

	remove_proc_entry("toshiba", 0);

	/* unregister the device file */

	misc_deregister(&tosh_device);

	/* release port 0xe4 and 0xe5 on older Toshiba laptops */

	if (tosh_fn==1)
		release_region(0xe4, 2);

	return;
}
#endif
