/* fan.c -- turn the internal fan on/off in a Toshiba Pentium(tm) laptop.
 *
 * Copyright (c) 1996,97,98  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: fan.c,v $
 * Revision 3.4  1999/12/01 19:59:23  jab
 * added daemon option to keep fan turned on when on external power
 * sorted out the assembler so it compiles and works whatever gcc is used
 *
 * Revision 3.3  1998/12/30 23:42:45  jab
 * added toggle option to turn the fan off if on and vice versa
 * started move to Hardware Configuration Interface routines
 *
 * Revision 3.2  1998/09/11 22:05:23  jab
 * switched to using routines from the System Configuration Interface
 * added GNU style long options
 * added auto option which turns the fan off if using battery and on if mains
 *
 * Revision 3.1  1998/08/22 10:33:27  jab
 * minor tidy ups to the code
 *
 * Revision 3.0  1998/07/11 14:01:38  jab
 * New selective switching method based on the return from GetMachineID,
 * should work on all known models
 *
 * Revision 2.2  1998/05/30 22:50:03  jab
 * hopefully fixed problems with reporting of fan status
 *
 * Revision 2.1  1998/05/08 22:52:17  jab
 * now drop root priveleges as soon as permision on the ports granted
 *
 * Revision 2.0  1998/01/31 16:00:23  jab
 * Changed to new method of turning the fan on/off, which should
 * work on a wider range of Toshiba laptops
 *
 * Revision 1.5  1997/05/23 13:17:25  jab
 * change the command line option processing to only deal with the first
 *
 * Revision 1.4  1997/04/29 21:26:11  jab
 * changed the name of the port variables to reflect their real status
 *
 * Revision 1.3  1996/08/01 14:25:36  jab
 * Added logging of changes in fan status via syslogd(8)
 *
 * Revision 1.2  1996/07/30 18:11:16  jab
 * fixed reporting of unknown command line options
 *
 * Revision 1.1  1996/06/25 21:47:42  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

static char const rcsid[]="$Id: fan.c,v 3.4 1999/12/01 19:59:23 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<unistd.h>
#include<syslog.h>
#include<signal.h>
#include<paths.h>
#include<pwd.h>
#include<features.h>
#include<sys/stat.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif

#include"hci.h"
#include"sci.h"

enum { ON, OFF, STATUS, AUTO, TOGGLE };

#define PID_FILE _PATH_VARRUN "fan.pid"


void FanOn(char *name)
{
	SMMRegisters reg;

	reg.eax = HCI_SET;
	reg.ebx = HCI_FAN;
	reg.ecx = HCI_ENABLE;
	HciFunction(&reg);

	syslog(LOG_INFO, "cooling fan turned on by %s", name);
	return;
}


void FanOff(char *name)
{
	SMMRegisters reg;

	reg.eax = HCI_SET;
	reg.ebx = HCI_FAN;
	reg.ecx = HCI_DISABLE;
	HciFunction(&reg);

	syslog(LOG_INFO, "cooling fan turned off by %s", name);
	return;
}


int FanStatus(void)
{
	SMMRegisters reg;

	reg.eax = HCI_GET;
	reg.ebx = HCI_FAN;
	HciFunction(&reg);

	if (reg.eax!=HCI_SUCCESS)
		return -1;
	else
		return (int) reg.ecx;
}


static void Catch(int sig)
{
	syslog(LOG_INFO, "Exiting" );
	unlink(PID_FILE);
	exit(0);
}


/*
 * Test to see if fan is already running in daemon mode
 */
void Running(void)
{
	int pid;
	FILE *str;

	/* check to see if fan daemon is already running */

	if (!access(PID_FILE, R_OK)) {
		if ((str = fopen(PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other daemon died unexpectedly */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "fan: daemon already running "
					"as process %d.\n", pid);
				exit(1);
			}

			fprintf(stderr, "fan: process %d appears to have "
				"died, continuing\n", pid);
			unlink(PID_FILE);
		}
	}
} 

/*
 * Enter daemon mode turning the fan on constantly while the mains is connected
 */
void Daemon(void)
{
	int pid;
	FILE *str;
	SMMRegisters reg;

	/* register some signal handlers */

	if (signal(SIGINT, SIG_IGN)!=SIG_IGN) signal(SIGINT, Catch);
	if (signal(SIGQUIT, SIG_IGN )!=SIG_IGN) signal(SIGQUIT, Catch);
	if (signal(SIGTERM, SIG_IGN)!=SIG_IGN) signal(SIGTERM, Catch);
	signal(SIGUSR1, SIG_IGN);

	/* parent */

	if ((pid=fork())) {
		if ((str=fopen(PID_FILE, "w"))) {
			fprintf(str, "%d\n", pid);
			fclose(str);
		}
		exit(0);
	}

	/* child */

	if (pid < 0) {
		syslog(LOG_INFO, "fork() failed: %m");
		unlink(PID_FILE);
		exit(1);
	} else {
		syslog(LOG_INFO, "entering daemon mode");
	}

	/* child - Follow the daemon rules in W. Richard Stevens.Advanced
	   Programming in the UNIX Environment (Addison-Wesley Publishing Co.,
	   1992). Page 417.). */

	if (setsid()<0) {
		syslog(LOG_INFO, "setsid() failed: %m");
		unlink(PID_FILE);
		exit(1);
	}

	chdir("/");
	umask(0);

	/* enter into an infinite loop */

	for (;;) {
		if (SciACPower()==SCI_MAINS) {
			reg.eax = HCI_SET;
			reg.ebx = HCI_FAN;
			reg.ecx = HCI_ENABLE;
			HciFunction(&reg);
		}
		sleep(1);
	}

}


int main(int argc, char *argv[])
{
	int opp;
	struct passwd *pw;
	char *name;


	/* check to see if already running in daemon mode */

	Running();

	/* check to make sure laptop has a fan */

	if (FanStatus()<0x00) {
		fprintf(stderr, "fan: laptop does not have cooling fan.\n");
		return 1;
	}

	/* open connection to system logger */

	openlog("fan", LOG_PID | LOG_CONS, LOG_USER);

	/* get user name */

	pw = getpwuid(getuid());
	name = pw ? pw->pw_name : getenv("LOGNAME");

	/* process command line arguments */

	opp = STATUS;
	if (argc>1) {
		if (!strcmp(argv[1], "-n") || !strcmp(argv[1], "--on")) {
			opp = ON;
		} else if (!strcmp(argv[1], "-f") || !strcmp(argv[1], "--off")) {
			opp = OFF;
		} else if (!strcmp(argv[1], "-a") || !strcmp(argv[1], "--auto")) {
			if (SciACPower()==SCI_BATTERY) {
				opp = OFF;
			} else {
				opp = ON;
			}
		} else if (!strcmp(argv[1], "-t") || !strcmp(argv[1], "--toggle")) {
			if (FanStatus()==0x00)
				opp = ON;
			else
				opp = OFF;
		} else if (!strcmp(argv[1], "-d") || !strcmp(argv[1], "--daemon")) {
			Daemon();
		} else {
			fprintf(stderr, "fan: illegal option %s\n"
				"Usage: fan [-n|f|a|t|d]\n", argv[0]);
			return 1;
		}
	}

	/* turn the fan on/off as requested */

	switch (opp) {
		case ON:
			FanOn(name);
			break;
		case OFF:
			FanOff(name);
			break;
	}

	/* Toshiba's fan.exe waits here for 0.1 seconds, so we do the same */
	
	usleep(100000);

	/* print the current status of the fan */

	if (FanStatus()==0x00)
		printf("Fan is off.\n");
	else
		printf("Fan is on.\n");

	return 0;
}
