/* thotswap.c -- enable hot swapping of IDE devices in the Selectbay
 *
 * Copyright (c) 1997-99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * Based on patches to hdparm by Christian Lademann <cal@zls.de> to use
 * the HDIO_UNREGISTER_HWIF and HDIO_SCAN_HWIF ioctls found in 2.2.x and
 * greater kernels.
 *
 * $Log:$
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static char const rcsid[]="$Id:$";


#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#include<fcntl.h>
#include<sys/stat.h>
#include<sys/wait.h>
#include<sys/utsname.h>
#include<errno.h>
#include<signal.h>
#include<syslog.h>
#include<paths.h>
#include<pwd.h>
#include<features.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif


#include"sci.h"
#include"hci.h"

#define PID_FILE "thotswap.pid"

static int id;
static int issued = 0;


#define PMAX 5
static long pid[PMAX];
static int retstatus[PMAX];
static int phead = 0;
static int ptail = 0;

#define BOOT_ALERT \
"Don't remove the device in the Selectable Bay.\n\
It's the boot drive. Please re-lock"


/*
 * Catch any signals and exit the program nicely
 */
void CatchSignal(int x)
{
	fprintf(stderr, "thotswap: caught signal %d, cleaning up...\n", x);

	SciCloseInterface();
	unlink(_PATH_VARRUN PID_FILE);
	exit(0);
}


/*
 * Handle SIGCHLD signals
 *
 * This function is called by the kernel when one of our child processes dies.
 * We need to call wait to get the status of the child and save it in a ring
 * buffer so it can be examined later. Before returning, we re-register the
 * signal handler.
 */
void NoZombie(int sig)
{
	long exited = 0;
	int status = 0;

	exited = wait(&status);

	if ((phead==ptail-1) || (phead==PMAX-1 && ptail==0)) {
		/* Ahhhhhhh. Fork bomb */
		return;
	} else {
		pid[phead] = exited;
		retstatus[phead] = status;
		phead = (phead==PMAX-1) ? 0 : ++phead;
	}

	signal(SIGCHLD, NoZombie);
	return;
}


/*
 * Install signal handlers so we can clean up nicely on an error
 */
void HandleSignals(void)
{
	signal(SIGHUP, CatchSignal);
	signal(SIGINT, CatchSignal);
	signal(SIGQUIT, CatchSignal);
	signal(SIGILL, CatchSignal);
	signal(SIGTRAP, CatchSignal);
	signal(SIGABRT, CatchSignal);
	signal(SIGIOT, CatchSignal);
	signal(SIGFPE, CatchSignal);
	signal(SIGKILL, CatchSignal);
	signal(SIGSEGV, CatchSignal);
	signal(SIGPIPE, SIG_IGN);
	signal(SIGTERM, CatchSignal);
	signal(SIGSTOP, CatchSignal);
	signal(SIGTTIN, CatchSignal);
	signal(SIGTTOU, CatchSignal);
	signal(SIGCHLD, NoZombie);
	signal(SIGUSR1, SIG_IGN);

	return;
}


/*
 * Use xmessage to display the battery alert messages
 */
int DisplayMessage(char *message, char *title)
{
	int pid,save,lead[2];

	/* create the pipe */

	if (pipe(lead)<0) {
		fprintf(stderr, "thotswap: unable to create pipe to "
			"xmessage\n");
		return -1;
	}

	if (fcntl(lead[0], F_SETFL, O_NDELAY)<0)
		fprintf(stderr, "thotswap: fatal fcntl call\n");

	pid = fork();
	if (pid>0) {
		save = dup(1);
		close(1);
		dup(lead[1]);
		close(lead[0]);
		close(lead[1]);
		fprintf(stdout, "%s\n", message);
		close(1);
		dup(save);
		return 127;
	} else if (pid==0) {
		char *argv[9] = {"xmessage", "-buttons", "Ok", "-title", title,
			"-center", "-file", "-", NULL};
		close(0);
		dup(lead[0]);
		close(lead[0]);
		close(lead[1]);
		execv(XMESSAGE, argv);
		return 127;
	}

	return -1;
}


/*
 * The Selectbay has been unlocked how do we need to proceed
 */
void SelectbayUnlocked(void)
{
	HciRegisters reg;
 	char message[128];

	/* is the in the device in the Selectbay an IDE/ATAPI device */

	reg.ax = HCI_GET;
	reg.bx = HCI_SELECT_STATUS;
	reg.cx = 0x0000;
	if (HciFunction(&reg)!=HCI_SUCCESS) {
		return;
	}

	if ((reg.cx==HCI_ATAPI) || (reg.cx==HCI_IDE)) {
		/* is the device in the Selectbay the boot device */

		reg.ax = HCI_GET;
		reg.bx = HCI_BOOT_DEVICE;
		reg.cx = HCI_SELECT_INT;
		HciFunction(&reg);
/*		if (HciFunction(&reg)!=HCI_SUCCESS) {
			printf("BBBB ax=0x%4x\n", reg.ax);
			return;
		}
*/
		if (reg.cx==HCI_ENABLE) {
			sprintf(message, BOOT_ALERT);
			DisplayMessage(message, "Caution!");
			return;
		}

		/* issue message to user to remove device */

		sprintf(message, BOOT_ALERT);
		DisplayMessage(message, "Remove device in the SelectBay");
	}
	
	return;
}



/*
 * Determine if configuration information has changed and needs to be re-read
 */
void HandleDeadChild(long pid, int status)
{
	/* I don't quite understand the values here but it works */

	if (status==25600) {
		/* reread the alarm settings */
	}

	if (status==51200) {
		/* set the battery save options */
	}

	return;
}


/*
 * This function is called from the main loop. It's job is to empty the
 * ring buffers filled by the SIGCHLD signal handler
 */
void ShovelDeadChildren()
{
	while (phead!=ptail) {
		HandleDeadChild(pid[ptail], retstatus[ptail]);
		ptail = (ptail==PMAX-1) ? 0 : ++ptail;
	}

	return;
}


/*
 * Enter daemon mode turning the fan on constantly while the mains is connected
 */
void Daemon(void)
{
	HciRegisters reg;
	int pid;
	FILE *str;

	/* register some signal handlers */

	if (signal(SIGINT, SIG_IGN)!=SIG_IGN) signal(SIGINT, CatchSignal);
	if (signal(SIGQUIT, SIG_IGN )!=SIG_IGN) signal(SIGQUIT, CatchSignal);
	if (signal(SIGTERM, SIG_IGN)!=SIG_IGN) signal(SIGTERM, CatchSignal);
	signal(SIGUSR1, SIG_IGN);

	/* parent */
/*
	if ((pid=fork())) {
		if ((str=fopen(_PATH_VARRUN PID_FILE, "w"))) {
			fprintf(str, "%d\n", pid);
			fclose(str);
		}
		exit(0);
	}
*/
	/* child */
/*
	if (pid<0) {
		syslog(LOG_INFO, "fork() failed: %m");
		unlink(_PATH_VARRUN PID_FILE);
		exit(1);
	} else {
		syslog(LOG_INFO, "entering daemon mode");
	}
*/
	/* child - Follow the daemon rules in W. Richard Stevens.Advanced
	   Programming in the UNIX Environment (Addison-Wesley Publishing Co.,
	   1992). Page 417.). */
/*
	if (setsid()<0) {
		syslog(LOG_INFO, "setsid() failed: %m");
		unlink(_PATH_VARRUN PID_FILE);
		exit(1);
	}

	chdir("/");
	umask(0);
*/
	/* loop until we die... */

	for (;;) {

		/* sleep for one second */

		sleep(1);

		/* has the SelectBay been unlocked? */

		reg.ax = HCI_GET;
		reg.bx = HCI_LOCK_STATUS;
		reg.cx = HCI_SELECT_INT;
		HciFunction(&reg);
		if (reg.cx==HCI_UNLOCKED) {
			SelectbayUnlocked();
			printf("1111\n");

		}

		/* handle dead child processes */

		ShovelDeadChildren();
	}

	return;
}



/*
 * the entry point of thotswap
 */
int main(int argc, char *argv[])
{
	int pid,version;
	int v1,v2,v3;
	FILE *str;
	struct utsname info;

	/* get the necessary I/O permissions */

	if (ioperm(0xb2, 1, 1)) {
		fprintf(stderr, "thotswap: can't get I/O permissions.\n");
		return 1;
	}

	/* do some quick checks on the laptop */

	if (SciSupportCheck(&version)==1) {
		fprintf(stderr, "thotswap: this computer is not supported\n");
		return 1;
	}

	if (HciGetMachineID(&id)==HCI_FAILURE) {
		fprintf(stderr, "thotswap: unable to get machine "
			"identification\n");
		return 1;
	}

	/* check that we are running a suitable kernel */

	if (uname(&info)) {
		fprintf(stderr, "thotswap: uname(): %s\n", strerror(errno));
		return 1;
	}

	sscanf(info.release, "%d.%d.%d", &v1, &v2, &v3);
	if (((v1*1000000)+(v2*1000)+v3)<2002000) {
		fprintf(stderr, "thotswap: can't run with linux < 2.2.0\n");
		return 1;
	}


        /* drop root priveleges to minimize the risk of running suid root */
/*
        seteuid(getuid());
        setegid(getgid());
*/

	/* check to see if a copy of thotswap is already running */

	if (!access(_PATH_VARRUN PID_FILE, R_OK)) {
		if ((str = fopen(_PATH_VARRUN PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other thotswap died unexpectedly */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "thotswap: already running as "
					"process %d.\n", pid);
				exit(1);
			}

			fprintf(stderr, "thotswap: process %d appears to have "
				"died, continuing\n", pid);
			unlink(_PATH_VARRUN PID_FILE);
		}
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(_PATH_VARRUN PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}

	/* install signal handling */

	HandleSignals();

	/* enter into daemon mode */

	Daemon();

	return 0;
}
