/* thotswap.c -- enable hot swapping of IDE devices in the Selectbay
 *
 * Copyright (c) 1997-2001  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * Based on patches to hdparm by Christian Lademann <cal@zls.de> to use
 * the HDIO_UNREGISTER_HWIF and HDIO_SCAN_HWIF ioctls found in 2.2.x and
 * greater kernels.
 *
 * Using beeps to warn the user idea came from David T-G <davidtg@bigfoot.com>,
 * based on code in the PCMCIA utilties Card Manager daemon by David A. Hinds
 * <dhinds@pcmcia.sourceforge.org>
 *
 * $Log: thotswap.c,v $
 * Revision 1.5  2001/02/02 17:20:23  jab
 * added notice about kernel module to error message
 * move the lock file when running as a daemon to /var/tmp
 *
 * Revision 1.4  2000/06/23 21:50:33  jab
 * test to make sure HciFunction call in main loop is successful before
 * trying to remove device from SelectBay
 *
 * Revision 1.3  2000/02/12 11:25:50  jab
 * switched to using waitpid to prevent possible race condition
 * fixed bug with not calling exit if execv failed in child processes
 *
 * Revision 1.2  1999/12/21 12:01:44  jab
 * added beeps to inform the user of errors/success
 *
 * Revision 1.1  1999/12/20 21:17:02  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static char const rcsid[]="$Id: thotswap.c,v 1.5 2001/02/02 17:20:23 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#include<fcntl.h>
#include<errno.h>
#include<signal.h>
#include<syslog.h>
#include<paths.h>
#include<mntent.h>
#include<sys/ioctl.h>
#include<sys/stat.h>
#include<sys/wait.h>
#include<sys/sysmacros.h>
#include<sys/utsname.h>
#include<sys/kd.h>
#include<linux/major.h>
#include<linux/hdreg.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif


#include"sci.h"
#include"hci.h"

#define PID_FILE _PATH_VARTMP ".thotswap.pid"

static int id;
static int locked = HCI_LOCKED;
static int removed = 0;
static int wall = 0;
static int xmessage = 0;
static int quite = 0;
static int console = 1;


#define BEEP_TIME 150
#define BEEP_OK   1000
#define BEEP_WARN 2000
#define BEEP_ERR  4000

#define VERSION_STRING "\
Toshiba HotSwap  version 1.2\n\
Copyright (c) 2000 Jonathan A. Buzzard <jonathan@buzzard.org.uk>\n"

#define USAGE \
"Usage: thotswap [OPTION]\n\n\
  -q,--quite       quite mode, don't beep when device is changed\n\
  -w,--wall        warn users of problems with wall\n\
  -x,--xmessage    warn users of problems with xmessage\n\
  -u,--unregister  try to unregister the device from the IDE driver\n\
  -r,--rescan      rescan the IDE bus to find devices\n\
  -h,--help        display this help message\n\
  -v,--version     display version\n\
Report bugs to jonathan@buzzard.org.uk\n"


/*
 * Catch any signals and exit the program nicely
 */
void CatchSignal(int x)
{
	fprintf(stderr, "thotswap: caught signal %d, cleaning up...\n", x);
	unlink(PID_FILE);
	exit(0);
}



/*
 * Determine if configuration information has changed and needs to be re-read
 */
void HandleDeadChild(long pid, int status)
{
	return;
}


/*
 * This function is called from the main loop. It's job is to find
 * dead children.
 */
void ShovelDeadChildren()
{
	pid_t childpid;
	int status;

	while ((childpid = waitpid(-1, &status, WNOHANG))>0) {
		HandleDeadChild(childpid, status);
 	}

	return;
}


/*
 * Use xmessage to display information messages
 */
int DisplayXMessage(char *message)
{
	int pid,lead[2];

	if (xmessage==0)
		return -1;

	/* create the pipe */

	if (pipe(lead)!=0) {
		fprintf(stderr, "thotswap: unable to create pipe to "
			"xmessage.\n");
		return -1;
	}

	if (fcntl(lead[0], F_SETFL, O_NDELAY)<0) {
		fprintf(stderr, "thotswap: fatal fcntl call.\n");
		return -1;
	}

	pid = fork();
	if (pid>0) {
		write(lead[1], message, strlen(message));
		close(lead[0]);
		close(lead[1]);
		return 127;
	} else if (pid==0) {
		char *argv[9] = {"xmessage", "-buttons", "Ok", "-title",
			"Toshiba HotSwap", "-center", "-file", "-", NULL};
		close(0);
		dup(lead[0]);
		close(lead[0]);
		close(lead[1]);
		execv(XMESSAGE, argv);
		exit(127);
	}

	return -1;
}


/*
 * Use syslog, wall and stderr as appropriate to display the messages
 */
int WriteMessage(char *message)
{
	int pid,lead[2];

	/* write message to stderr or log via syslog */

	if (console==0)
		fprintf(stderr, "%s", message);
	else
		syslog(LOG_INFO, message);

	/* only continue if wall messages are required */

	if (wall==0)
		return 127;

	/* create the pipe */

	if (pipe(lead)!=0) {
		fprintf(stderr, "thotswap: unable to create pipe to wall.\n");
		return -1;
	}

	if (fcntl(lead[0], F_SETFL, O_NDELAY)<0) {
		fprintf(stderr, "thotswap: fatal fcntl call.\n");
		return -1;
	}

	pid = fork();
	if (pid>0) {
		write(lead[1], "thotswap: ", 10);
		write(lead[1], message, strlen(message));
		close(lead[0]);
		close(lead[1]);
		return 127;
	} else if (pid==0) {
		char *argv[2] = {"wall", NULL};
		close(0);
		dup(lead[0]);
		close(lead[0]);
		close(lead[1]);
		execv(WALL, argv);
		exit(127);
	}

	return -1;
}


/*
 * Make the PC speaker produce a beep
 */
void Beep(unsigned int ms, unsigned int freq)
{
	int fd, arg;

	if (quite==1)
		return;

	fd = open("/dev/console", O_RDWR);
	if (fd < 0)
		return;

	arg = (ms << 16) | freq;
	ioctl(fd, KDMKTONE, arg);
	close(fd);
	usleep(ms*1000);

	return;
}


/*
 * Does the device in the SelectBay hold a mounted filesystem
 */
int DeviceMounted(void)
{
	FILE *str;
	struct mntent *info;
	int mounted;

	mounted = 0;
	str = setmntent(_PATH_MOUNTED, "r");

	/* loop though the mounted file systems */

	info = getmntent(str);
	while (info!=NULL) {
		if (strncmp("/dev/hdc", info->mnt_fsname, 8)==0) {
			mounted = 1;
			break;
		}
		info = getmntent(str);
	}

	endmntent(str);

	return mounted;
}


/*
 * Try and unregister the IDE device in the SelectBay
 */
int SelectbayUnregister(void)
{
	int fd,loop,err;
	struct stat buf;

	/* check the device exists */

	if (stat("/dev/hda", &buf)) {
		return -1;
	}

	/* make sure this is IDE channel 0 */

	if (major(buf.st_rdev)!=IDE0_MAJOR) {
		return -1;
	}

	/* open the device for sending commands */

	fd = open("/dev/hda", O_RDONLY);
	if (fd<0) {
		return -1;
	}

	/* request a device in the SelectBay is unregistered */

	for (loop=0;loop<2;loop++) {
		err = ioctl(fd, HDIO_UNREGISTER_HWIF, 1);
		if (err==0)
			break;
		sleep(1);
	}

	if (err==-1) {
		Beep(BEEP_TIME, BEEP_ERR);
		syslog(LOG_INFO, "WARNING: unable to unregister the device in "
				"the SelectBay, please re-lock\n");
		DisplayXMessage("WARNING: unable to unregister\n        "
			"the device in the SelectBay.\n         "
			"Please re-lock");
	}

	close (fd);

	return err;
}


/*
 * Force a rescan of the IDE bus to find any devices in the SelectBay
 */
int SelectbayRescan(void)
{
	int fd,loop,err;
	int args[]={0x170, 0, 15};
	struct stat buf;

	if (stat("/dev/hda", &buf)) {
		return -1;
	}

	if (major(buf.st_rdev)!=IDE0_MAJOR) {
		return -1;
	}

	fd = open("/dev/hda", O_RDONLY);
	if (fd<0) {
		return -1;
	}

	/* loop as sometimes it fails for no apparent reason */

	for (loop=0;loop<2;loop++) {
		err = ioctl(fd, HDIO_SCAN_HWIF, args);
		if (err==0)
			break;
		sleep(1);
	}

	if (err==-1) {
		Beep(BEEP_TIME, BEEP_WARN);
		WriteMessage("WARNING: error rescanning the IDE bus.\n");
		DisplayXMessage("WARNING: error rescanning the IDE bus");
	}
	
	Beep(BEEP_TIME, BEEP_OK);
	WriteMessage("device succesfully inserted into SelectBay.");
	DisplayXMessage("Device succesfully inserted into SelectBay.");
	close(fd);
	return err;
}


/*
 * The Selectbay has been unlocked how do we need to proceed
 */
void SelectbayUnlocked(void)
{
	SMMRegisters reg;

	/* check we don't already have had a removed device */

	if (removed==1)
		return;

	/* make sure the device has no mounted file systems */

	if (DeviceMounted()==1) {
		Beep(BEEP_TIME, BEEP_ERR);
		WriteMessage("WARNING: device in SelectBay contains "
				"mounted filesystems, please re-lock.\n");
		DisplayXMessage("WARNING: device in SelectBay contains\n"
			"         mounted filesystems. Please re-lock.");
		return;
	}

	/* is the in the device in the Selectbay an IDE/ATAPI device */

	reg.eax = HCI_GET;
	reg.ebx = HCI_SELECT_STATUS;
	reg.ecx = 0x0000;
	if (HciFunction(&reg)!=HCI_SUCCESS) {
		return;
	}

	if ((reg.ecx==HCI_ATAPI) || (reg.ecx==HCI_IDE)) {
		/* is the device in the Selectbay the boot device */

		reg.eax = HCI_GET;
		reg.ebx = HCI_BOOT_DEVICE;
		reg.ecx = HCI_SELECT_INT;
		if (HciFunction(&reg)!=HCI_SUCCESS) {
			return;
		}

		if (reg.ecx==HCI_ENABLE) {
			Beep(BEEP_TIME, BEEP_ERR);
			WriteMessage("WARNING: device in SelectBay is boot "
				"device, please re-lock.\n");
			DisplayXMessage("WARNING: device in SelectBay is boot\n"
				"         device. Please re-lock.");
			return;
		}

		/* attempt to remove device from IDE bus */

		if (SelectbayUnregister()<0) {
			Beep(BEEP_TIME, BEEP_ERR);
			WriteMessage("WARNING: unable to remove device in the "
				"SelectBay, please re-lock.\n");
			DisplayXMessage("WARNING:  unable to remove device in\n"
				"         the SelectBay. Please re-lock.");
			return;
		}

		Beep(BEEP_TIME, BEEP_OK);
		WriteMessage("device in the SelectBay removed.\n");
		DisplayXMessage("Device in the SelectBay sucessfully removed.");
		removed = 1;
	}
	
	return;
}


/*
 * The Selectbay has been locked try and rescan IDE bus.
 */
void SelectbayLocked(void)
{

	/* have we previously removed a device successfully? */

	if (removed==0)
		return;

	if (SelectbayRescan()<0) {
		return;
	}

	removed = 0;
	return;
}


/*
 * Enter daemon mode scanning the Selectbay every second
 */
void Daemon(void)
{
	SMMRegisters reg;
	int pid;
	FILE *str;

	/* register some signal handlers */

	if (signal(SIGINT, SIG_IGN)!=SIG_IGN) signal(SIGINT, CatchSignal);
	if (signal(SIGQUIT, SIG_IGN )!=SIG_IGN) signal(SIGQUIT, CatchSignal);
	if (signal(SIGTERM, SIG_IGN)!=SIG_IGN) signal(SIGTERM, CatchSignal);
	signal(SIGPIPE, SIG_IGN);
	signal(SIGUSR1, SIG_IGN);

	/* parent */

	if ((pid=fork())) {
		if ((str=fopen(PID_FILE, "w"))) {
			fprintf(str, "%d\n", pid);
			fclose(str);
		}
		exit(0);
	}

	/* child */

	if (pid<0) {
		syslog(LOG_INFO, "fork() failed: %m");
		unlink(PID_FILE);
		exit(1);
	} else {
		syslog(LOG_INFO, "entering daemon mode");
	}

	/* child - Follow the daemon rules in W. Richard Stevens.Advanced
	   Programming in the UNIX Environment (Addison-Wesley Publishing Co.,
	   1992). Page 417.). */

	if (setsid()<0) {
		syslog(LOG_INFO, "setsid() failed: %m");
		unlink(PID_FILE);
		exit(1);
	}

	chdir("/");
	umask(0);

	/* loop until we die... */

	for (;;) {

		/* sleep for one second */

		sleep(1);

		/* has the SelectBay been unlocked? */

		reg.eax = HCI_GET;
		reg.ebx = HCI_LOCK_STATUS;
		reg.ecx = HCI_SELECT_INT;
		if (HciFunction(&reg)==HCI_SUCCESS) {
			if ((reg.ecx==HCI_UNLOCKED) && (locked==HCI_LOCKED)) {
				syslog(LOG_INFO, "SelectBay unlocked");
				locked = HCI_UNLOCKED;
				SelectbayUnlocked();
			} else if ((reg.ecx==HCI_LOCKED) && (locked==HCI_UNLOCKED)) {
				syslog(LOG_INFO, "SelectBay locked");
				locked = HCI_LOCKED;
				SelectbayLocked();
			}
		}

		/* handle dead child processes */

		ShovelDeadChildren();
	}

	return;
}


/*
 * exit thotswap
 */
void Quit(int status)
{
	unlink(PID_FILE);
	exit(status);
}


/*
 * process the command line arguments
 */
void ProcessComandLine(int *argc, char ***argv)
{
	int i,rescan,unregister;

	rescan = 0;
	unregister = 0;
	for (i=1;i<*argc;i++) {
		if ((!strcmp((*argv)[i], "-h")) || (!strcmp((*argv)[i], "--help"))) {
			printf(USAGE);
			Quit(0);
		} else if ((!strcmp((*argv)[i], "-v")) || (!strcmp((*argv)[i], "--version"))) {
			printf(VERSION_STRING);
			Quit(0);
		} else if ((!strcmp((*argv)[i], "-w")) || (!strcmp((*argv)[i], "--wall"))) {
			wall = 1;
		} else if ((!strcmp((*argv)[i], "-q")) || (!strcmp((*argv)[i], "--quite"))) {
			quite = 1;
		} else if ((!strcmp((*argv)[i], "-x")) || (!strcmp((*argv)[i], "--xmessage"))) {
			xmessage = 1;
                } else if ((!strcmp((*argv)[i], "-u")) || (!strcmp((*argv)[i], "--unregister"))) {
                        unregister = 1;
                } else if ((!strcmp((*argv)[i], "-r")) || (!strcmp((*argv)[i], "--rescan"))) {
			rescan = 1;
                } else {
			fprintf(stderr, "thotswap: unrecognised option %s\n",
				(*argv)[i]);
			Quit(1);
		}
	}

	/* carry out the command line rescaning and removing if requested */

	if ((rescan==1) && (unregister==0)) {
		console = 1;
		removed = 1;
		SelectbayLocked();
		Quit(0);
	} else if ((rescan==0) && (unregister==1)) {
		console = 1;
		SelectbayUnlocked();
		Quit(0);
	} else if ((rescan==1) && (unregister==1)) {
		fprintf(stderr, "thotswap: can't rescan and unregister at the "
			"same time");
		Quit(1);
	}

	return;
}


/*
 * the entry point of thotswap
 */
int main(int argc, char *argv[])
{
	int pid,version;
	int v1,v2,v3;
	FILE *str;
	struct utsname info;


	/* this program *must* be run as root */

	if (getuid()) {
		fprintf(stderr, "thotswap: must be run as root.\n" );
		return 1;
	}

	/* do some quick checks on the laptop */

	if (SciSupportCheck(&version)==1) {
		fprintf(stderr, "thotswap: this computer is not supported "
			"or the kernel module not installed.\n");
		return 1;
	}

	if (HciGetMachineID(&id)==HCI_FAILURE) {
		fprintf(stderr, "thotswap: unable to get machine "
			"identification\n");
		return 1;
	}

	/* check that we are running a suitable kernel */

	if (uname(&info)) {
		fprintf(stderr, "thotswap: uname(): %s\n", strerror(errno));
		return 1;
	}

	sscanf(info.release, "%d.%d.%d", &v1, &v2, &v3);
	if (((v1*1000000)+(v2*1000)+v3)<2002000) {
		fprintf(stderr, "thotswap: can't run with linux < 2.2.0\n");
		return 1;
	}

	/* check to see if a copy of thotswap is already running */

	if (!access(PID_FILE, R_OK)) {
		if ((str = fopen(PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other thotswap died unexpectedly */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "thotswap: already running as "
					"process %d.\n", pid);
				exit(1);
			}

			fprintf(stderr, "thotswap: process %d appears to have "
				"died, continuing\n", pid);
			unlink(PID_FILE);
		}
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}

	/* open connection to system logger */

	openlog("thotswap", LOG_PID | LOG_CONS, LOG_USER);

	/* process command line arguments */

	ProcessComandLine(&argc, &argv);

	/* enter into daemon mode */

	Daemon();

	return 0;
}
