/* wmtuxtime.c -- WindowMaker dockable applet portion of TuxTime
 *
 * Copyright (c) 1997-99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log:$
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static char const rcsid[]="$Id:$";


#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#include<sys/stat.h>
#include<errno.h>
#include<signal.h>
#include<paths.h>
#include<pwd.h>
#include<features.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif
#include<X11/X.h>
#include<X11/xpm.h>


#include"wmtuxtime.h"
#include"wmgeneral.h"
#include"wmtuxtime.xpm"
#include"wmtuxtime_mask.xbm"
#include"sci.h"
#include"hci.h"
#include"config.h"


enum { EMPTY, PRIMARY, AUXILLARY };

int id;
int power = SCI_MAINS;
int selectbay = EMPTY;
int alarms;
char *config;
static BatteryState low,critical;


#define COMMAND "echo -e \'%s\' | xmessage -buttons Ok -center -file -"


/*
 * Catch any signals and exit the program nicely
 */
void CatchSignal(int x)
{
	fprintf(stderr, "wmtuxtime: caught signal %d, cleaning up...\n", x);

	SciCloseInterface();
	unlink(_PATH_VARRUN PID_FILE);
	exit(0);
}


/*
 * Install signal handlers so we can clean up nicely on an error
 */
void HandleSignals(void)
{
	signal(SIGHUP, CatchSignal);
	signal(SIGINT, CatchSignal);
	signal(SIGQUIT, CatchSignal);
	signal(SIGILL, CatchSignal);
	signal(SIGTRAP, CatchSignal);
	signal(SIGABRT, CatchSignal);
	signal(SIGIOT, CatchSignal);
	signal(SIGFPE, CatchSignal);
	signal(SIGKILL, CatchSignal);
	signal(SIGSEGV, CatchSignal);
	signal(SIGPIPE, CatchSignal);
	signal(SIGTERM, CatchSignal);
	signal(SIGSTOP, CatchSignal);
	signal(SIGTTIN, CatchSignal);
	signal(SIGTTOU, CatchSignal);
	signal(SIGUSR1, SIG_IGN);

	return;
}


/*
 * Trigger a software suspend sequence using the HCI
 */
void SoftwareSuspend(void)
{
	HciRegisters reg;

	reg.ax = HCI_SET;
	reg.bx = HCI_SOFTWARE_SUSPEND;
	reg.cx = HCI_ENABLE;

	HciFunction(&reg);

	return;
}


/*
 * Handle button presses
 */
void pressEvent(XButtonEvent *event)
{	
	int x,y;
	char message[128],line[192];
	
	x = event->x;
	y = event->y;

	if(x>=6 && y>=48 && x<=18 && y<=58) {
		/* next arrow button */

	 	copyXPMArea(66, 44, 13, 11, 6, 48);
		if ((selectbay!=EMPTY) && (power==SCI_BATTERY)) {
			switch (selectbay) {
				case PRIMARY:
					copyXPMArea(79, 93, 15, 10, 44, 33);
					selectbay = AUXILLARY;
					break;
			 	case AUXILLARY:
					copyXPMArea(96, 93, 15, 10, 44, 33);
					selectbay = BOTH;
					break;
				case BOTH:
					copyXPMArea(62, 93, 15, 10, 44, 33);
					selectbay = PRIMARY;
					break;
			}
		}
		RedrawWindow();
		printf("BUTTON: next\n");
		usleep(20000L);
	} else if (x>=19 && y>=48 && x<=31 && y<=58){
		/* alarm settings */

		copyXPMArea(79, 44, 13, 11, 19, 48);
		RedrawWindow();
		printf("BUTTON: alarms\n");
		sprintf(message, TIME_BATTERY_ALERT, 20);
		sprintf(line, COMMAND, message);
		printf("%s\n", line);
		system(line);

		usleep(20000L);
	} else if (x>=32 && y>=48 && x<=44 && y<=58){
		/* suspend call */

		copyXPMArea(92, 44, 13, 11, 32, 48);
		RedrawWindow();
		printf("BUTTON: suspend\n");
		SoftwareSuspend();
		usleep(20000L);
	} else if (x>=45 && y>=48 && x<=57 && y<=58){
		/* machine settings */

		copyXPMArea(105, 44, 13, 11, 45, 48);
		RedrawWindow();
		system("/home/jab/c/toshutils/src/tuxtime");
		printf("BUTTON: settings\n");
		usleep(20000L);
	}

	return;
}


/*
 * Convert a single character to an interger, similar to atoi 
 */
inline int ctoi(char *s)
{
	return ((*s>='0') && (*s<='9')) ? *s-'0' : -1;
}


/*
 * Convert the time string to hours and minutes in the 24 hour format
 */
int ConvertTime(char *time, int *hour, int *minute)
{
	/* we can't rely on short circuit boolean evaluation */

	if (*time=='\0')
		return 0;
	if (*(time+1)=='\0')
		return 0;
	if (*(time+2)=='\0')
		return 0;

	/* have we a dropped leading zero? */

	if (*(time+1)==':') {
		*hour = ctoi(time);
		time+=2;
	}
	else if (*(time+2)!=':')
		return 0;
	else {
		*hour = (ctoi(time+1))+(10*ctoi(time));
		time+=3;
	}

	if (*time=='\0')
		return 0;
	if (*(time+1)=='\0')
		return 0;

	*minute = ctoi(time+1)+(10*ctoi(time));

	/* deal with any am or pm switch */

	if (*(time+2)!='\0')
		if ((*(time+2)=='p') || (*(time+2)=='P'))
			*hour+=12;
		else if ((*(time+2)!='h') && (*(time+2)!='H') &&
			(*(time+2)!='a') && (*(time+2)!='A'))
			return 0;

	/* check to see if the converted time is valid */

	if ((*hour>=0) && (*hour<24) && (*minute>=0) && (*minute<60))
		return 1;
	else
		return 0;
}


/*
 * Create the default .tuxtimerc file at the given location
 */
void CreateDefaultRC(char *file)
{
	WriteConfigString("TuxTime", "Left", "300", file);
	WriteConfigString("TuxTime", "Top", "200", file);
	WriteConfigString("TuxTime", "Title", "1", file);
	WriteConfigString("TuxTime", "Iconic", "0", file);
	WriteConfigString("TuxTime", "AlarmPowerOn", "12:00H", file);
	WriteConfigString("TuxTime", "PollInterval", "10", file);
	WriteConfigString("TuxTime", "AutoSetCharge", "1", file);

	WriteConfigString("Display", "Size", "1", file);
	WriteConfigString("Display", "Modules", "1", file);
	WriteConfigString("Display", "Style", "0", file);
	WriteConfigString("Display", "Icon", "1", file);
	WriteConfigString("Display", "Cycle", "0", file);
	WriteConfigString("Display", "CycleButton", "0", file);
	WriteConfigString("Display", "CycleInterval", "15", file);
	WriteConfigString("Display", "CycleCrnt", "1", file);
	WriteConfigString("Display", "Title", "Justify", file);
	WriteConfigString("Display", "RightOffset", "0", file);
	WriteConfigString("Display", "LeftOffset", "0", file);
	WriteConfigString("Display", "ShowAllBatteries", "0", file);

	WriteConfigString("BatPowerSettings", "HDDAutoOff", "8", file);
	WriteConfigString("BatPowerSettings", "DisplayAutoOff", "8", file);
	WriteConfigString("BatPowerSettings", "SleepMode", "1", file);
	WriteConfigString("BatPowerSettings", "CPUSpeed", "0", file);
	WriteConfigString("BatPowerSettings", "LCDBrightness", "1", file);
	WriteConfigString("BatPowerSettings", "BatterySaveMode", "2", file);
	WriteConfigString("BatPowerSettings", "SystemAutoOff", "512", file);
	WriteConfigString("BatPowerSettings", "CoolingMethod", "1", file);

	WriteConfigString("ExtPowerSettings", "HDDAutoOff", "8", file);
	WriteConfigString("ExtPowerSettings", "DisplayAutoOff", "8", file);
	WriteConfigString("ExtPowerSettings", "SleepMode", "1", file);
	WriteConfigString("ExtPowerSettings", "CPUSpeed", "0", file);
	WriteConfigString("ExtPowerSettings", "LCDBrightness", "2", file);
	WriteConfigString("ExtPowerSettings", "BatterySaveMode", "0", file);
	WriteConfigString("ExtPowerSettings", "SystemAutoOff", "512", file);
	WriteConfigString("ExtPowerSettings", "CoolingMethod", "1", file);

	WriteConfigString("AlarmLow", "Percent", "20", file);
	WriteConfigString("AlarmLow", "Time", "00:20", file);
	WriteConfigString("AlarmLow", "Trigger", "1", file);

	WriteConfigString("AlarmCritical", "Percent", "10", file);
	WriteConfigString("AlarmCritical", "Time", "00:10", file);
	WriteConfigString("AlarmCritical", "Trigger", "1", file);

	return;
}


/*
 * Setup the alarm values by reading them in from the config file
 */
void SetupAlarms(char *config)
{
	char buffer[8];

	low.percent = GetConfigInt("AlarmLow", "Percent", 20, config);
	low.trigger = GetConfigInt("AlarmLow", "Trigger", 3, config);
	GetConfigString("AlarmLow", "Time", "00:20", buffer, 8, config);
	ConvertTime(buffer, &(low.hour), &(low.minute));
	critical.percent = GetConfigInt("AlarmCritical", "Percent", 10, config);
	critical.trigger = GetConfigInt("AlarmCritical", "Trigger", 3, config);
	GetConfigString("AlarmCritical", "Time", "00:10", buffer, 8, config);
	ConvertTime(buffer, &(critical.hour), &(critical.minute));

	return;
}


/*
 * Test to see what condition the battery is in (i.e. normal, low, critial)
 */
int BatteryTest(BatteryState *settings, int percent, int time)
{
	int flag;

	flag = 0;
	if ((settings->trigger==TIME) || (settings->trigger==BOTH))
		if (time<=MINUTES(settings->hour, settings->minute))
			flag |= TIME;
			
	if ((settings->trigger==PERCENT) || (settings->trigger==BOTH))
		if (percent<=settings->percent)
			flag |= PERCENT;

	return flag;
}


/*
 * Change the power saving settings to reflect the power source
 */
void PowerChanged(int source, char *file)
{
	unsigned short save,hdd,display,sleep;
	unsigned short speed,lcd,autooff,cooling;
	SciRegisters reg;
	int i;
	char *section;

	/* read the settings from the config file */

        if (source==SCI_MAINS)
                section = "ExtPowerSettings";
        else
                section = "BatPowerSettings";

        /* get the settings from the configuration file */

        save = GetConfigInt(section, "BatterySaveMode", 2, file);
	hdd = GetConfigInt(section, "HDDAutoOff", 512, file);
	display = GetConfigInt(section, "DisplayAutoOff", 1, file);
	sleep = GetConfigInt(section, "SleepMode", 0, file);
	speed = GetConfigInt(section, "CPUSpeed", 1, file);
	lcd = GetConfigInt(section, "LCDBrightness", 0, file);
	autooff = GetConfigInt(section, "SystemAutoOff", 1, file);
	cooling = GetConfigInt(section, "CoolingMethod", 0, file);

	SciOpenInterface();

	/* try and set the battery save mode */

	reg.setting = SCI_BATTERY_SAVE;
	reg.current = save;
	if (SciSet(&reg)==SCI_SUCCESS) {
		i = 0;
	} else {
		i = 1;
	}

	printf("SAVEMODE=%d\n", i);

	/* set the other settings if user mode or new style machine */

	if (save==SCI_USER_SETTINGS) {
	}

	SciCloseInterface();

	return;
}


/*
 * Display the battery alarm messages using xmessage or if avaliable
 * wmessage
 *
 */
void DisplayAlarm(int state, int percent, int time)
{
	int trigger,hour,minute;
 	char message[128],line[192];

	if (state==EMERGENCY) {
		sprintf(message, EMERGENCY_BATTERY_ALERT);
		sprintf(line, COMMAND, message);
		system(line);
		return;
	}

	hour = SCI_HOUR(time);
	minute = SCI_MINUTE(time);
	trigger = NEVER;


	/* determine what triggered the alarm */

	if (state==LOW)
		trigger = BatteryTest(&low, percent, MINUTES(hour, minute));
	else if (state==CRITICAL)
		trigger = BatteryTest(&critical, percent, MINUTES(hour, minute));

	/* display the correct alarm message */

	if (trigger & TIME) {
		sprintf(message, TIME_BATTERY_ALERT, MINUTES(hour, minute));
		sprintf(line, COMMAND, message);
		system(line);
	}

	if (trigger & PERCENT) {
		sprintf(message, PERCENT_BATTERY_ALERT, percent);
		sprintf(line, COMMAND, message);
		system(line);
	}
	


	return;
}



/*
 * Update the battery life remaining display, and switch settings if the
 * power source has changed
 */
void Update(void)
{
	SciRegisters reg;
	int time,percent,second,battery,hour,minute,state,i,j;

	time = 0;
	percent = 0;

	/* get the percent and time remaining */

	SciOpenInterface();
	reg.setting = SCI_BATTERY_PERCENT;
	if (SciGet(&reg)==SCI_SUCCESS)
		percent = ((100*reg.current)/reg.possible);
	else
		percent = -1;

	reg.setting = SCI_BATTERY_TIME;
	if (SciGet(&reg)==SCI_SUCCESS)
		time = reg.current;
	else
		time = -1;

	/* check to see if their is a battery in the SelectBay */

	reg.setting = SCI_2ND_BATTERY;
	if (SciGet(&reg)==SCI_SUCCESS) {
		if (reg.current!=0xffff)
			second = ((100*reg.current)/reg.possible);
		else
			second = -1;
		if ((power==SCI_BATTERY) && (selectbay==EMPTY))
			copyXPMArea(62, 93, 15, 10, 44, 33);
		if (selectbay==EMPTY)
			selectbay = PRIMARY;
	} else {
		selectbay = EMPTY;
		second = -1;
	}

	SciCloseInterface();

	/* calculate the battery condition */

	state = NORMAL;
	hour = SCI_HOUR(time);
	minute = SCI_MINUTE(time);
	if ((percent<3) && (percent>=0))
		state = EMERGENCY;
	else if (BatteryTest(&critical, percent, MINUTES(hour, minute))!=0)
		state = CRITICAL;
	else if (BatteryTest(&low, percent, MINUTES(hour, minute))!=0)
		state = LOW;

	/* send out the battery alarms */

	if ((state==EMERGENCY) && ((alarms & EMERGENCY)==0)) {
		alarms |= EMERGENCY;
		DisplayAlarm(state, percent, time);
	}
	if ((state==LOW) && ((alarms & LOW)==0)) {
		fprintf(stderr,"LOW BATTERY ALARM\n");	
		alarms |= LOW;
		DisplayAlarm(state, percent, time);
	}
	if ((state==CRITICAL) && ((alarms & CRITICAL)==0)) {
		fprintf(stderr,"LOW BATTERY ALARM\n");	
		alarms |= CRITICAL;
		DisplayAlarm(state, percent, time);
	}

	/* turn off alarm states as necessary */

	if (state==NORMAL)
		alarms = NORMAL;
	else if (state==LOW)
		alarms = LOW & NORMAL;
	else if (state==CRITICAL)
		alarms = CRITICAL & LOW & NORMAL;

	/* check AC status. */

	if (SciACPower()!=power) {
		if (power==SCI_MAINS) {
			power = SCI_BATTERY;
			if (selectbay==EMPTY) {
				copyXPMArea(45, 93, 15, 10, 44, 33);
			} else {
				selectbay = PRIMARY;
				copyXPMArea(62, 93, 15, 10, 44, 33);
			}
			PowerChanged(power, config);
			printf("BATTERY\n");
		} else {
			power = SCI_MAINS;
			if (selectbay!=EMPTY)
				selectbay = BOTH;
			copyXPMArea(28, 93, 15, 10, 44, 33);
			PowerChanged(power, config);
			printf("MAINS\n");
		}
	}

	/* paste up the default charge status and time */

	copyXPMArea(104,  6,  5, 7, 6, 7);
	copyXPMArea(83, 81, 41, 9, 15, 7);

	/* check to see if we are charging */

/*	if ((int)(my_cur_info.battery_status)==3) {
		copyXPMArea(98,  6, 5, 7,  6,  7);
		copyXPMArea(75, 81, 1, 2, 17,  9);
		copyXPMArea(75, 81, 1, 2, 17, 12);
	}*/

	/* repaint buttons. */

	copyXPMArea(66, 56, 52, 11, 6, 48);

	/* paste up the "Time Left", this time means:
	     If not charging: Time left before battery drains to 0% 
	     If charging:     Time left before battery gets to maximum */

	if (time<0) {
		/* dodgy time so just display a "null" indicator (--:--) */

		copyXPMArea(83, 81, 41, 9, 15, 7);
	} else {
		copyXPMArea((hour/10)*7+5, 81, 7, 9, 21, 7);
		copyXPMArea((hour%10)*7+5, 81, 7, 9, 29, 7);
		copyXPMArea(76, 81, 2, 9, 38, 7);
		copyXPMArea((minute/10)*7+5, 81, 7, 9, 42, 7);
		copyXPMArea((minute%10)*7+5, 81, 7, 9, 50, 7);
	}
	
	/*  do battery percentage */

	switch (selectbay) {
		case EMPTY: case PRIMARY:
			battery = percent;
			break;
		case AUXILLARY:
			battery = second;
			break;
		case BOTH:
			battery = (percent+second)/2;
			break;
		default:
			battery = percent;
			break;
	}

	copyXPMArea(5, 93, 19, 7, 7, 34);   /* Show Default % */
	copyXPMArea(66, 4, 49, 9, 7, 21);    /* Show Default Meter */
	if (battery==100) {
		copyXPMArea(15, 71, 1, 7,  7, 34);    /* If 100%, show 100% */
		copyXPMArea(5, 71, 6, 7,  9, 34);
		copyXPMArea(5, 71, 6, 7, 15, 34);
		copyXPMArea(65, 71, 7, 7, 21, 34);    /* Show '%' */
		copyXPMArea(66, 14, 49, 9, 7, 21);    /* Show Meter */
	} else if (battery>=0) {
		if (battery>=10)
			copyXPMArea((battery/10)*6+4, 71, 6, 7, 9, 34);
		copyXPMArea((battery%10)*6+4, 71, 6, 7, 15, 34);
		copyXPMArea(65, 71, 7, 7, 22, 34);    /* Show '%' */

		/* Show Meter */

		i = battery*49/100;
		switch (state) {
			case NORMAL:
				j = 14;
				break;
			case LOW:
				j = 24;
				break;
			case CRITICAL:
				j = 34;
				break;
			default:
				j = 14;
		}

		copyXPMArea(66, j, i, 9, 7, 21);
	}

		
	return;
}


/*
 * the entry point of wmTuxTime
 */
int main(int argc, char *argv[])
{

	int pid,version,bios,poll,i;
	XEvent event;
	FILE *str;
	struct passwd *pw;
	struct stat info;

	/* get the necessary I/O permissions */

	if (ioperm(0xb2, 1, 1)) {
		fprintf(stderr, "tuxtime: can't get I/O permissions.\n");
		return 1;
	}

	/* development hack */
	SciCloseInterface();

	/* do some quick checks on the laptop */

	if (SciSupportCheck(&version)==1) {
		fprintf(stderr, "wmtuxtime: this computer is not supported\n");
		return 1;
	}

	bios = HciGetBiosVersion();
	if (bios==0) {
		fprintf(stderr, "wmtuxtime: unable to get BIOS version\n");
		return 1;
	}

	if (HciGetMachineID(&id)==HCI_FAILURE) {
		fprintf(stderr, "wmtuxtime: unable to get machine "
			"identification\n");
		return 1;
	}

	/* check to see if a copy of wmTuxTime is already running */

	if (!access(_PATH_VARRUN PID_FILE, R_OK)) {
		if ((str = fopen(_PATH_VARRUN PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other TuxTime died unexpectedly */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "wmtuxtime: already running as "
					"process %d.\n", pid);
				exit(1);
			}

			fprintf(stderr, "tuxtime: process %d appears to have "
				"died, continuing\n", pid);
			unlink(_PATH_VARRUN PID_FILE);
		}
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(_PATH_VARRUN PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}

	/* get the path of the tuxtimerc file */

	pw = getpwuid(getuid());
	config = (char *) malloc(strlen(pw->pw_dir)+12);
	if (config==NULL) {
		fprintf(stderr, "tuxtime: unable to allocate sufficent memory, "
			"exiting\n");
		exit(1);
		}
	strcpy(config, pw->pw_dir);
	strcat(config, "/.tuxtimerc");

	/* test to see if tuxtimerc file exists and if no create one */

	if ((stat(config, &info)!=0) && (errno==ENOENT)) {
		CreateDefaultRC(config);
	}

	/* install signal handling */

	HandleSignals();

	/* read the config file and initilize state*/

	SetupAlarms(config);
	alarms = NORMAL;
	if (SciACPower()==SCI_MAINS)
		power = SCI_BATTERY;
	else
		power = SCI_MAINS;
	poll = GetConfigInt("TuxTime", "PollInterval", 10, config);

	/* open the X11 windows */

	openXwindow(argc, argv, wmtuxtime_xpm, wmtuxtime_mask_bits,
		wmtuxtime_mask_width, wmtuxtime_mask_height);

	/* loop until we die... */

	i = 0;
	for (;;) {

		/*  only process battery state every poll cycles of this loop.
		    We run it faster to catch the expose events, etc... */ 

		if (i>poll) {
			Update();
			i = 0;
		} else {
			i++;
		}

		/* process any pending X events */

		while (XPending(display)) {
			XNextEvent(display, &event);
			switch(event.type){
				case Expose:
					RedrawWindow();
					break;
				case ButtonPress:
					pressEvent(&event.xbutton);
					break;
				case ButtonRelease:
                        		break;
			}
		}

		/* Redraw and wait for next update  */

		RedrawWindow();
		usleep(DELAY);
	}

	return 0;
}
