/* tuxtime.c -- entry point for TuxTime
 *
 * Copyright (c) 1996/99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: tuxtime.c,v $
 * Revision 1.4  1999/05/25 08:13:16  jab
 * moved display updated routines to seperate file
 * other minor cleaning of the code
 *
 * Revision 1.3  1998/08/18 08:44:09  jab
 * use the TuxTime icon, thanks to Patrick Reyonlds <patrick@cs.virgina.edu>
 * plus other small code tidy ups
 *
 * Revision 1.2  1998/08/18 08:04:32  jab
 * inital command line interface implemented
 *
 * Revision 1.1  1998/08/09 09:12:31  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

static const char rcsid[]="$Id: tuxtime.c,v 1.4 1999/05/25 08:13:16 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#include<sys/stat.h>
#include<errno.h>
#include<signal.h>
#include<paths.h>
#include<pwd.h>
#include<features.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif
#include<gtk/gtk.h>

#include"sci.h"
#include"hci.h"
#include"misc.h"
#include"machine.h"
#include"alarms.h"
#include"display.h"
#include"update.h"
#include"config.h"

#include"tuxtime.xpm"

#define USAGE \
"USAGE STRING TO GO HERE\n"

#define VERSION_STRING "\
TuxTime Beta version "VERSION"\n\
Copyright (c) 1996,1997,1998,1999 Jonathan A. Buzzard <jab@hex.prestel.co.uk>\n"

int id;
int libretto;
char model[8];
char *config;

struct Arguments {
	char *value;
	unsigned short setting;
};

struct Options {
	char *opt;
	char *longopt;
	unsigned short mode;
	int models;
	struct Arguments possible[10];
};

enum {
	MODEL_UNKNOWN = 0x0000,
	MODEL_FC2F    = 0x0001,
	MODEL_FCCA =    0x0002,
	MODEL_FCCB    = 0x0004,
	MODEL_FCCC    = 0x0008,
	MODEL_ALL     = 0xffff
};

#define ENABLE_DISABLE {\
	{ "enabled", SCI_ENABLED },\
	{ "disabled", SCI_DISABLED },\
	{ NULL, 0 }\
}

struct Options opts[] = {
	{"-s", "--sleep", SCI_SLEEP_MODE, MODEL_ALL, ENABLE_DISABLE },
	{"-p", "--panel", SCI_PANEL_POWER, MODEL_ALL, ENABLE_DISABLE },
	{"-a", "--alarm", SCI_ALARM_POWER, MODEL_ALL, {
		{"disable", SCI_ALARM_DISABLED}, {"#time", SCI_ALARM_ENABLED}, {NULL,
		0}}},
	{NULL, NULL, 0, 0, {{NULL, 0}}}
};


/*
 * Deal with window manager delete events
 */
void Deleted(GtkWidget *widget, GdkEvent *event, gpointer *data)
{
	unlink(_PATH_VARRUN PID_FILE);
	gtk_main_quit();
}


/*
 * Catch any signals and exit the program nicely
 */
void CatchSignal(int x)
{
	fprintf(stderr, "tuxtime: caught signal %d, cleaning up...\n", x);

	SciCloseInterface();
	unlink(_PATH_VARRUN PID_FILE);
	exit(0);
}


/*
 * Process the non-GTK command line arguments
 */
void ProcessComandLine(int *argc, char ***argv)
{
	int i,j,k,quit;
	int hours,minutes;


	quit = 0;
	for (i=1;i<*argc;i++) {
		if ((!strcmp((*argv)[i], "-h")) || (!strcmp((*argv)[i], "--help"))) {
			printf(USAGE);
			unlink(_PATH_VARRUN PID_FILE);
			exit(0);
		}
		if ((!strcmp((*argv)[i], "-v")) || (!strcmp((*argv)[i], "--version"))) {
			printf(VERSION_STRING);
			unlink(_PATH_VARRUN PID_FILE);
			exit(0);
		}

		/* set through our command line argument structure */

		for (j=0;opts[j].opt;j++) {

			printf("%s %s %s\n",(*argv)[i], opts[j].opt, opts[j].longopt);

			if ((!strcmp(opts[j].opt, (*argv)[i])) ||
				(!strcmp(opts[j].longopt, (*argv)[i]))) {
				
				/* is this feature supported on this model? */
				
				if (id & opts[j].models) {
					for (k=0;opts[j].possible[k].value;k++) {

						/* do we have the optional argument? */
						
						printf("%d %d %s\n",i+1, *argc, (*argv)[i+1]); 

						if (((i+1)<*argc) && ((*argv)[i+1])) {
							if (!strcmp(opts[j].possible[k].value, (*argv)[i+1])) {
								SetSetting(opts[j].mode,
									opts[j].possible[k].setting, ERROR_CONSOLE);
								i++;
								quit++;
							}
							if (!strcmp(opts[j].possible[k].value, "#time")) {

								if (ConvertTime((*argv)[i], &hours, &minutes)==1) {
									SetSetting(opts[j].mode,
									SCI_ALARM_ENABLED | SCI_TIME(hours, minutes),
										ERROR_CONSOLE);
									i++;
									quit++;
								}
								else {
									printf("tuxtime: illegal setting for alarm power on time\n");
									unlink(_PATH_VARRUN PID_FILE);
									exit(1);
								}
							}
						}
						else {
							fprintf(stderr, "tuxtime: missing required setting option\n");
							unlink(_PATH_VARRUN PID_FILE);
							exit(1);
						}
					}
				}
				else {
					fprintf(stderr, "tuxtime: feature not supported on this "
						"model\n");
					unlink(_PATH_VARRUN PID_FILE);
					exit(1);
				}
			}
		}
		if (!opts[j].opt) {
			fprintf(stderr, "tuxtime: unrecognised option %s\n", (*argv)[i]);
			unlink(_PATH_VARRUN PID_FILE);
			exit(1);
		}

	}

	/* exit if requested */

	if (quit>0) {
		unlink(_PATH_VARRUN PID_FILE);
		exit(0);
	}

	return;
}


/*
 * If we recieve a signal, exit nicely freeing up resources
 */
void HandleSignals(void)
{
	signal(SIGHUP, CatchSignal);
	signal(SIGINT, CatchSignal);
	signal(SIGQUIT, CatchSignal);
	signal(SIGILL, CatchSignal);
	signal(SIGTRAP, CatchSignal);
	signal(SIGABRT, CatchSignal);
	signal(SIGIOT, CatchSignal);
	signal(SIGFPE, CatchSignal);
	signal(SIGKILL, CatchSignal);
	signal(SIGSEGV, CatchSignal);
	signal(SIGPIPE, CatchSignal);
	signal(SIGTERM, CatchSignal);
	signal(SIGCHLD, CatchSignal);
	signal(SIGSTOP, CatchSignal);
	signal(SIGTTIN, CatchSignal);
	signal(SIGTTOU, CatchSignal);
	signal(SIGUSR1, SIG_IGN);

	return;
}


/*
 * Create the menu for the main window
 */
void AddMenu(GtkWidget *menubar)
{
	GtkWidget *menu,*menuitem;
	GtkWidget *submenu,*subitem;


	menu = gtk_menu_new();
	submenu = gtk_menu_new();

	menuitem = gtk_menu_item_new_with_label("Display...");
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
		GTK_SIGNAL_FUNC(DisplayDialog), NULL);
	menuitem = gtk_menu_item_new_with_label("Alarms...");
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
		GTK_SIGNAL_FUNC(AlarmDialog), NULL);
	menuitem = gtk_menu_item_new_with_label("Set Battery Charge...");
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_widget_set_sensitive(GTK_WIDGET(menuitem), FALSE);
	menuitem = gtk_menu_item_new_with_label("Machine Settings...");
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
		GTK_SIGNAL_FUNC(MachineDialog), NULL);
	menuitem = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu), menuitem);

	menuitem = gtk_menu_item_new_with_label("Help");
	gtk_menu_append(GTK_MENU(menu), menuitem);

	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), submenu);

	subitem = gtk_menu_item_new_with_label("Contents");
	gtk_menu_append(GTK_MENU(submenu), subitem);
	gtk_widget_set_sensitive(GTK_WIDGET(subitem), FALSE);
	subitem = gtk_menu_item_new_with_label("Search for Help On...");
	gtk_menu_append(GTK_MENU(submenu), subitem);
	gtk_widget_set_sensitive(GTK_WIDGET(subitem), FALSE);
	subitem = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(submenu), subitem);
	subitem = gtk_menu_item_new_with_label("How to Use Help");
	gtk_menu_append(GTK_MENU(submenu), subitem);
	gtk_widget_set_sensitive(GTK_WIDGET(subitem), FALSE);
	subitem = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(submenu), subitem);
	subitem = gtk_menu_item_new_with_label("About TuxTime...");
	gtk_menu_append(GTK_MENU(submenu), subitem);
	gtk_signal_connect(GTK_OBJECT(subitem), "activate",
		GTK_SIGNAL_FUNC(AboutCallback), NULL);

	menuitem = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu), menuitem);

	menuitem = gtk_menu_item_new_with_label("Exit");
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
		GTK_SIGNAL_FUNC(ExitCallback), NULL);

	menuitem = gtk_menu_item_new_with_label("Options");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menu);
	gtk_menu_bar_append(GTK_MENU_BAR(menubar), menuitem);

	return;
}


/*
 * the entry point of TuxTime
 */
int main(int argc, char *argv[])
{
	int pid,version,flag,bios,poll;
	FILE *str;
	struct passwd *pw;
	struct stat info;
	GtkWidget *window,*box,*menubar;
	GtkStyle *style;
	GdkPixmap *icon;
	GdkBitmap *mask;


	/* get the necessary I/O permissions */

	if (ioperm(0xb2, 1, 1)) {
		fprintf(stderr, "tuxtime: can't get I/O permissions.\n");
		return 1;
	}

	SciCloseInterface();

	/* do some quick checks on the laptop */

	flag = SciSupportCheck(&version);
	if (flag==1) {
		fprintf(stderr, "tuxtime: this computer is not supported\n");
		return 1;
	}

	bios = HciGetBiosVersion();
	if (bios==0) {
		fprintf(stderr, "tuxtime: unable to get BIOS version\n");
		return 1;
	}

	if (HciGetMachineID(&id)==HCI_FAILURE) {
		fprintf(stderr, "tuxtime: unable to get machine identification\n");
		return 1;
	}

	if (SciGetModel(model)==SCI_FAILURE) {
		fprintf(stderr, "tuxtime: unable to get model information\n");
	}

	switch (id) {
		case 0xfc2f: /* New style ID mess */
		case 0xfcc3: /* Texcra 720x */
		case 0xfcca: /* Satellite Pro 400x */
		case 0xfccb: /* Portage 610CT */
		case 0xfccc: /* Tecra 700x */
		case 0xfcdd: /* Satellite 110x */
		case 0xfcdf: /* Tecra 500x */
			printf("tuxtime: machine id : 0x%04x    BIOS version : "
				"%d.%d    SCI version: %d.%d\n", id,
				(bios & 0xff00)>>8, bios & 0xff,
				(version & 0xff00)>>8, version & 0xff);
			break;
		default:
			printf("tuxtime: unrecognized machine identification:\n");
			printf("machine id : 0x%04x    BIOS version : %d.%d    "
				"model : %s\n", id, (bios & 0xff00)>>8, bios & 0xff, model);
			printf("\nplease email this information to jonathan@buzzard.org.uk ");
			printf("and include the model\nand model number, eg. Libretto 50CT");
			printf(" model number PA1249E\n");
	}

	/* try and determine if this machine is a Libretto */

	libretto = 0;
	if (id==0xfc2f) {
		if (!strcmp(model, "50CT"))
			libretto = 1;
		else if (!strcmp(model, "70CT"))
			libretto = 1;
	}

	/* check to see if a copy of TuxTime is already running */

	if (!access(_PATH_VARRUN PID_FILE, R_OK)) {
		if ((str = fopen(_PATH_VARRUN PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other TuxTime died unexpectedly */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "tuxtime: already running as "
					"process %d.\n", pid);
				exit(1);
			}

			fprintf(stderr, "tuxtime: process %d appears to have "
				"died, continuing\n", pid);
			unlink(_PATH_VARRUN PID_FILE);
		}
	}

	/* get the path of the tuxtimerc file */

	pw = getpwuid(getuid());
	config = (char *) malloc(strlen(pw->pw_dir)+12);
	if (config==NULL) {
		fprintf(stderr, "tuxtime: unable to allocate sufficent memory, "
			"exiting\n");
		exit(1);
		}
	strcpy(config, pw->pw_dir);
	strcat(config, "/.tuxtimerc");

	/* test to see if tuxtimerc file exists and if no create one */

	if ((stat(config, &info)!=0) && (errno==ENOENT)) {
		CreateDefaultRC(config);
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(_PATH_VARRUN PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}

	/* initialize the GTK toolkit */

	gtk_init(&argc, &argv);

	/* install signal handling after GTK toolkit is initialize otherwise
	   it will override our signal handling */

	HandleSignals();

	/* Gtk options now cleared, process remaining command line arguments */

	ProcessComandLine(&argc, &argv);

	/* create the top level window */

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_container_border_width(GTK_CONTAINER(window), 1);
	gtk_signal_connect(GTK_OBJECT(window), "delete_event",
		GTK_SIGNAL_FUNC(Deleted), NULL);

	/* add window tile and icon */

	gtk_window_set_title(GTK_WINDOW(window), "TuxTime");
	style = gtk_widget_get_style(window);
	gtk_widget_show(window);
	icon = gdk_pixmap_create_from_xpm_d(window->window, &mask,
		&style->bg[GTK_STATE_NORMAL], (gchar **)tuxtime_xpm);
	gdk_window_set_icon(window->window, NULL, icon, mask);

	/* add menu bar */

	box = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(window), box);

	menubar = gtk_menu_bar_new();
	gtk_box_pack_start(GTK_BOX(box), menubar, FALSE, TRUE, 0);
	AddMenu(menubar);

	/* add the modules for showing battery life */

	progress =  gtk_progress_bar_new();
	gtk_box_pack_start(GTK_BOX(box), progress, FALSE, TRUE, 0);

	percent = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(box), percent, FALSE, TRUE, 0);

	if (!libretto) {
		time = gtk_label_new("");
		gtk_box_pack_start(GTK_BOX(box), time, FALSE, TRUE, 0);
	}

	/* initalize the alarm variables */

	SetupAlarms();

	/* call the Update function to set the display */

	Update(NULL);

	gtk_widget_show_all(window);

	/* add the timeout for the update function */

	poll = GetConfigInt("TuxTime", "PollInterval", 10, config);
	gtk_timeout_add(poll*1000, Update, (gpointer) 1);

	/* start the program */

	gtk_main();

	return 0;
}
