/* misc.c -- some miscellaneous functions for TuxTime
 *
 * Copyright (c) 1996-99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: misc.c,v $
 * Revision 1.3  1998/08/16 16:57:53  jab
 * added alarm time fuctions and updated settings stuff for error message type
 *
 * Revision 1.2  1998/08/15 23:10:37  jab
 * ok button on error dialogs fixed, thanks to Keith Packard <keithp@ncd.com>
 *
 * Revision 1.1  1998/08/13 23:02:57  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

static const char rcsid[]="$Id: misc.c,v 1.3 1998/08/16 16:57:53 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<unistd.h>
#include<paths.h>
#include<gtk/gtk.h>

#include"sci.h"
#include"misc.h"
#include"machine.h"
#include"alarms.h"
#include"display.h"
#include"config.h"

#include"tuxtime.xpm"
#include"warning.xpm"


/*
 * Callback to destroy error and about dialog boxes when Ok is pressed
 */
void DialogCallback(GtkWidget *widget, gpointer *data)
{
	gtk_widget_destroy(GTK_WIDGET(data));
	return;
}


/*
 * Print the SCI error on the console
 */
void ConsoleError(int type, char *message)
{
	switch (type) {
		case 0:
			fprintf(stderr, "tuxtime: error could not open "
				"interface to BIOS\n");
			break;
		case 1:
			fprintf(stderr, "tuxtime: error could not set BIOS\n");
			break;
		case 2:
			fprintf(stderr, "tuxtime: error could not get BIOS "
				"setting\n");
			break;
		default:
			fprintf(stderr, "tuxtime: unknown error\n");
	}

	return;
}


/*
 * Generate an error dialog describing the current error type
 */
void GuiError(int type, char *message)
{
	GtkWidget *window,*button,*label,*box,*pixmap;
	GtkStyle *style;
	GdkPixmap *icon;
	GdkBitmap *mask;
	char string[256];

	window = gtk_dialog_new();

	/* handle the window manager close signal */

	gtk_signal_connect(GTK_OBJECT(window), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &window);

	button = gtk_button_new_with_label("Ok");
	gtk_widget_set_usize(button, 60, 20);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(DialogCallback), GTK_OBJECT(window));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->action_area), button,
		FALSE, FALSE, 0);

	/* draw the message area */

	box = gtk_hbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), box,
		TRUE, TRUE, 0);

	style = gtk_widget_get_style(window);
	icon = gdk_pixmap_create_from_xpm_d(window->window, &mask,
		&style->bg[GTK_STATE_NORMAL], (gchar **)warning_xpm);
	pixmap = gtk_pixmap_new(icon, mask);
	gtk_box_pack_start(GTK_BOX(box), pixmap, FALSE, FALSE, 4);

	switch (type) {
		case ERROR_OPEN:
			label = gtk_label_new("ERROR: could not open interface"
				" to hardware");
			break;
		case ERROR_SET:
			sprintf(string, "ERROR: could not set %s", message);
			label = gtk_label_new(string);
			break;
		case ERROR_GET:
			sprintf(string, "ERROR: could not get %s", message);
			label = gtk_label_new(string);
			break;
		case ERROR_MESSAGE:
			sprintf(string, "%s", message);
			label = gtk_label_new(string);
			break;
		default:
			label = gtk_label_new("ERROR: unknown");
	}

	gtk_misc_set_padding(GTK_MISC(label), 10, 10);
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, TRUE, 0);

	gtk_widget_show_all(window);

	return;
}


/*
 * Generate an string describing the mode
 */
char *GetErrorMessage(unsigned short mode)
{
	switch (mode) {
		case SCI_BATTERY_SAVE:
			return "Battery save mode";
		case SCI_DISPLAY_AUTO:
			return "Display auto off time";
		case SCI_HDD_AUTO_OFF:
			return "Hard disk auto off time";
		case SCI_SYSTEM_AUTO:
			return "System auto off time";
		case SCI_LCD_BRIGHTNESS:
			return "LCD Brightness setting";
		case SCI_COOLING_METHOD:
			return "Cooling method";
		case SCI_LCD_BACKLIGHT:
			return "LCD Backlight setting";
		default:
			return "unrecognised setting";
	}
}


/*
 * Get the current, possible and default values of a setting complete with
 * opening and closing of the interface to the system configuration interface
 */
int GetSetting(SciRegisters *reg, int errtype)
{
	int error,setting;

	setting = reg->setting;

	error = SciOpenInterface();
	if (error!=SCI_SUCCESS) {
		if (errtype==ERROR_CONSOLE)
			ConsoleError(0, NULL);
		else
			GuiError(0, NULL);
		return error;
	}

	error = SciGet(reg);
	if (error!=SCI_SUCCESS) {
		if (errtype==ERROR_CONSOLE)
			ConsoleError(2, GetErrorMessage(setting));
		else
			GuiError(2, GetErrorMessage(setting));
		SciCloseInterface();
		return error;
	}

	SciCloseInterface();

	return SCI_SUCCESS;
}


/*
 * Set the setting of a mode, complete with opening and closing of the
 * interface to the system management mode
 */
int SetSetting(unsigned short mode, unsigned short status, int errtype)
{
	SciRegisters reg;
	int error,setting;

	reg.setting = mode;
	reg.current = status;
	setting = reg.setting;

	error = SciOpenInterface();
	if (error!=SCI_SUCCESS) {
		if (errtype==ERROR_CONSOLE)
			ConsoleError(0, NULL);
		else
			GuiError(0, NULL);
		return error;
	}

	error = SciSet(&reg);
	if (error!=SCI_SUCCESS) {
		if (errtype==ERROR_CONSOLE)
			ConsoleError(2, GetErrorMessage(setting));
		else
			GuiError(2, GetErrorMessage(setting));
		SciCloseInterface();
		return error;
	}

	SciCloseInterface();

	return SCI_SUCCESS;
}


/*
 * Close down a dialog
 */
void OkCallback(GtkWidget *widget, gpointer *data)
{
	gtk_widget_destroy(GTK_WIDGET(data));
	return;
}

/*
 * Exit TuxTime
 */
void ExitCallback(GtkWidget *widget, GdkEventButton *event, gpointer *data)
{
	unlink(_PATH_VARRUN PID_FILE);
	gtk_main_quit();
}


/*
 * Quit a dialog without changing the settings
 */
static void CancelCallback(GtkWidget *widget, GtkWidget *dialog)
{
	gtk_widget_destroy(dialog);
	dialog = NULL;
	return;
}


/*
 * Pop up an about dialog box
 */
void AboutCallback(GtkWidget *widget, GdkEventButton *event, gpointer *data)
{
	GtkWidget *window,*button,*label,*box,*icon;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GdkColor *transparent = NULL;

	window = gtk_dialog_new();

	/* handle the window manager close signal */

	gtk_signal_connect(GTK_OBJECT(window), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &window);

	/* add the Ok button */

	button = gtk_button_new_with_label("Ok");
	gtk_widget_set_usize(button, 60, 20);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(DialogCallback), GTK_OBJECT(window));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->action_area), button,
		FALSE, FALSE, 0);

	/* draw the message area */

	box = gtk_hbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), box,
		TRUE, TRUE, 0);

	pixmap = gdk_pixmap_create_from_xpm_d(window->window, &mask,
		transparent, (gchar **)tuxtime_xpm);
	icon = gtk_pixmap_new(pixmap, mask);
	gtk_box_pack_start(GTK_BOX(box), icon, FALSE, FALSE, 4);

	label = gtk_label_new("TuxTime Version "VERSION"\n"
		"(c) 1996-99 Jonathan Buzzard.");
	gtk_misc_set_padding(GTK_MISC(label), 10, 10);
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, TRUE, 0);

	gtk_widget_show_all(window);

	return;
}


/*
 * Add the four buttons that appear on the right of each page
 */
void AddButtons(GtkWidget *box, GtkWidget *window, int type)
{
	GtkWidget *ok,*cancel,*defaults,*help;

	ok = gtk_button_new_with_label("Ok");
	gtk_widget_set_usize(ok, 60, 20);
	gtk_box_pack_start(GTK_BOX(box), ok, FALSE, FALSE, 0);

	cancel = gtk_button_new_with_label("Cancel");
	gtk_widget_set_usize(cancel, 60, 20);
	gtk_box_pack_start(GTK_BOX(box), cancel, FALSE, FALSE, 0);

	defaults = gtk_button_new_with_label("Defaults");
	gtk_widget_set_usize(defaults, 60, 20);
	gtk_box_pack_start(GTK_BOX(box), defaults, FALSE, FALSE, 0);

	help = gtk_button_new_with_label("Help");
	gtk_widget_set_usize(help, 60, 20);
	gtk_box_pack_start(GTK_BOX(box), help, FALSE, FALSE, 0);
	gtk_widget_set_sensitive(GTK_WIDGET(help), FALSE);

	switch(type) {
		case BUTTONS_ALARM:
			gtk_signal_connect(GTK_OBJECT(ok), "clicked",
				GTK_SIGNAL_FUNC(SaveAlarmSettingsCallback),
				GTK_WINDOW(window));
			gtk_signal_connect(GTK_OBJECT(cancel), "clicked",
				GTK_SIGNAL_FUNC(AlarmCancelCallback),
				GTK_OBJECT(window));
			gtk_signal_connect(GTK_OBJECT(defaults), "clicked",
				GTK_SIGNAL_FUNC(SetAlarmDefaultsCallback),
				NULL);
			break;
		case BUTTONS_DISPLAY:
			gtk_signal_connect(GTK_OBJECT(ok), "clicked",
				GTK_SIGNAL_FUNC(SaveDisplaySettingsCallback),
				GTK_OBJECT(window));
			gtk_signal_connect(GTK_OBJECT(cancel), "clicked",
				GTK_SIGNAL_FUNC(CancelCallback),
				GTK_OBJECT(window));
			gtk_signal_connect(GTK_OBJECT(defaults), "clicked",
				GTK_SIGNAL_FUNC(SetDisplayDefaultsCallback),
				NULL);
			break;
		case BUTTONS_SETTING:
			gtk_signal_connect(GTK_OBJECT(ok), "clicked",
				GTK_SIGNAL_FUNC(MachineOkCallback),
				GTK_OBJECT(window));
			gtk_signal_connect(GTK_OBJECT(cancel), "clicked",
				GTK_SIGNAL_FUNC(MachineCancelCallback),
				GTK_OBJECT(window));
			gtk_signal_connect(GTK_OBJECT(defaults), "clicked",
				GTK_SIGNAL_FUNC(MachineDefaultsCallback), NULL);
			break;
	}

	return;
}


/*
 * Convert a single character to an interger, similar to atoi 
 */
inline int ctoi(char *s)
{
	return ((*s>='0') && (*s<='9')) ? *s-'0' : -1;
}


/*
 * Convert the time string to hours and minutes in the 24 hour format
 */
int ConvertTime(char *time, int *hour, int *minute)
{
	/* we can't rely on short circuit boolean evaluation */

	if (*time=='\0')
		return 0;
	if (*(time+1)=='\0')
		return 0;
	if (*(time+2)=='\0')
		return 0;

	/* have we a dropped leading zero? */

	if (*(time+1)==':') {
		*hour = ctoi(time);
		time+=2;
	}
	else if (*(time+2)!=':')
		return 0;
	else {
		*hour = (ctoi(time+1))+(10*ctoi(time));
		time+=3;
	}

	if (*time=='\0')
		return 0;
	if (*(time+1)=='\0')
		return 0;

	*minute = ctoi(time+1)+(10*ctoi(time));

	/* deal with any am or pm switch */

	if (*(time+2)!='\0')
		if ((*(time+2)=='p') || (*(time+2)=='P'))
			*hour+=12;
		else if ((*(time+2)!='h') && (*(time+2)!='H') &&
			(*(time+2)!='a') && (*(time+2)!='A'))
			return 0;

	/* check to see if the converted time is valid */

	if ((*hour>=0) && (*hour<24) && (*minute>=0) && (*minute<60))
		return 1;
	else
		return 0;
}

/*
 * Create the default .tuxtimerc file at the given location
 */
void CreateDefaultRC(char *file)
{
	WriteConfigString("TuxTime", "Left", "300", file);
	WriteConfigString("TuxTime", "Top", "200", file);
	WriteConfigString("TuxTime", "Title", "1", file);
	WriteConfigString("TuxTime", "Iconic", "0", file);
	WriteConfigString("TuxTime", "AlarmPowerOn", "12:00H", file);
	WriteConfigString("TuxTime", "PollInterval", "10", file);
	WriteConfigString("TuxTime", "AutoSetCharge", "1", file);

	WriteConfigString("Display", "Size", "1", file);
	WriteConfigString("Display", "Modules", "1", file);
	WriteConfigString("Display", "Style", "0", file);
	WriteConfigString("Display", "Icon", "1", file);
	WriteConfigString("Display", "Cycle", "0", file);
	WriteConfigString("Display", "CycleButton", "0", file);
	WriteConfigString("Display", "CycleInterval", "15", file);
	WriteConfigString("Display", "CycleCrnt", "1", file);
	WriteConfigString("Display", "Title", "Justify", file);
	WriteConfigString("Display", "RightOffset", "0", file);
	WriteConfigString("Display", "LeftOffset", "0", file);
	WriteConfigString("Display", "ShowAllBatteries", "0", file);

	WriteConfigString("BatPowerSettings", "HDDAutoOff", "8", file);
	WriteConfigString("BatPowerSettings", "DisplayAutoOff", "8", file);
	WriteConfigString("BatPowerSettings", "SleepMode", "1", file);
	WriteConfigString("BatPowerSettings", "CPUSpeed", "0", file);
	WriteConfigString("BatPowerSettings", "LCDBrightness", "1", file);
	WriteConfigString("BatPowerSettings", "BatterySaveMode", "2", file);
	WriteConfigString("BatPowerSettings", "SystemAutoOff", "512", file);
	WriteConfigString("BatPowerSettings", "CoolingMethod", "1", file);

	WriteConfigString("ExtPowerSettings", "HDDAutoOff", "8", file);
	WriteConfigString("ExtPowerSettings", "DisplayAutoOff", "8", file);
	WriteConfigString("ExtPowerSettings", "SleepMode", "1", file);
	WriteConfigString("ExtPowerSettings", "CPUSpeed", "0", file);
	WriteConfigString("ExtPowerSettings", "LCDBrightness", "2", file);
	WriteConfigString("ExtPowerSettings", "BatterySaveMode", "0", file);
	WriteConfigString("ExtPowerSettings", "SystemAutoOff", "512", file);
	WriteConfigString("ExtPowerSettings", "CoolingMethod", "1", file);

	WriteConfigString("AlarmLow", "Percent", "20", file);
	WriteConfigString("AlarmLow", "Time", "00:20", file);
	WriteConfigString("AlarmLow", "Trigger", "1", file);

	WriteConfigString("AlarmCritical", "Percent", "10", file);
	WriteConfigString("AlarmCritical", "Time", "00:10", file);
	WriteConfigString("AlarmCritical", "Trigger", "1", file);

	return;
}
