/* external.c - call the external processes of TuxTime
 *
 * Copyright (C) 1998 Steve Borho <borho@stlnet.com>
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "mount.h"

#define     P_MAX      5
static long pid[P_MAX];
static int  retStatus[P_MAX];
static int  p_head = 0, p_tail = 0;


typedef struct PInfo
{
	int childPid;           /* pid of child  */
	int childStatus;        /* status of child */
	char childStatusChar;   /* status of child as char to display */
}
PInfo;
   



void NewSystem(char *command, PInfo * pinf)
{
	int pid = 0;

	if (pinf && pinf->childPid)
		return;

	pid = fork();
	if (pid==0) {
		char *argv[4] = {"sh", "-c", command, 0};
		execv("/bin/sh", argv);
		exit(127);
	} else if ((pid!=-1) && pinf) {
		pinf->childPid = pid;
	}
	return;
}


void CallConfigApp(allInfo *ai)
{
	char buf[256];

	if (!fork()) {
		char *argv[5] = {"config", "-s", "-m", ai->mpi[ai->curMount]->mountStr, 0};

		sprintf(buf, "%s/mount.conf", BINDIR);
		execv(buf, argv);
		exit(127);
	}
	return;
}


void ShowHelpPage()
{
    char buf[256];

    if (!fork())
    {
        char           *argv[3] =
        {"config", "-H", 0};

        sprintf(buf, "%s/mount.conf", BINDIR);

        execv(buf, argv);
        exit(127);
    }
}


void HandleReturnCode(PInfo *pinf)
{
    char filebuf[256];
    char *filename = (char *) 0;
    char *title = (char *) 0;

    switch(WEXITSTATUS(pinf->childStatus))
    {
    case 2:
        /*
         * Insufficient rights to mount device
         */
        mp->childStatusChar = 'P';
        filename = "noperms.txt";
        title = "Not enough permissions";
        break;
    case 1:
        /*
         * Device busy
         */
        mp->childStatusChar = 'B';
        filename = "devicebusy.txt";
        title = "Drive in use";
        break;
    case 32:
        /*
         * Not ready or directory already in use
         */
        mp->childStatusChar = 'R';
        filename = "notmountable.txt";
        title = "Device not ready";
        break;
    }

    if (aiPtr->newbieMessages && filename != (char *) 0)
    {
        sprintf(filebuf, "%s/%s", ICONDIR, filename);

        if (!fork())
        {
            char           *argv[8] =
            {"xmessage", "-name", title, "-file", filebuf, "-buttons",
                "I gotchya", 0};

            execv(XMESSAGE, argv);
            exit(127);
        }
    }
}

void HandleDeadChild(long pid, int status)
{
	PInfo *pinf = 0;
	int i;

	for (i = 0; i < aiPtr->numMounts ; i++) {
		if (aiPtr->mpi[i]->childPid == pid) {
			mp = aiPtr->mpi[i];
			break;
		}
	}
   
	if (pinf) {
		pinf->childPid = 0;
		pinf->childStatus = status;
		pinf->childStatusChar = 0;

        if (i == aiPtr->curMount)
        {
            aiPtr->currentMountChanged = TRUE;
        }
        if (WIFEXITED(status))
        {
            if (WEXITSTATUS(status))
            {
                aiPtr->errorOccured = TRUE;
            }
            else if (mp->performEject)
            {
                mp->performEject = FALSE;
                ejectDrive(mp);
            }
        }
    }
}


/*
 * Handle SIGCHLD signals
 *
 * This function is called by the kernel when one of our child
 * processes dies.  We need to call wait to get the status of
 * the child and save it in a ring buffer so it can be examined
 * later. Before returning, we re-register the signal handler.
 */
void noZombie(int sig)
{
	long exitedPid = 0;
	int status = 0;

	exitedPid = wait(&status);

	if ((p_head==p_tail-1) || (p_head==P_MAX-1 && p_tail==0)) {
		/* Ahhhhhhh. Fork bomb */
		return;
	} else {
		pid[p_head] = exitedPid;
		retStatus[p_head] = status;
		p_head = (p_head == P_MAX-1) ? 0 : ++p_head;
	}

	signal(SIGCHLD, noZombie);
	return;
}


/*
 * This function is called from the main loop. It's job is to
 * empty the ring buffers filled by the SIGCHLD signal handler
 */
void shovelDeadChildren()
{
	while (p_head != p_tail) {
		handleDeadChild(pid[p_tail], retStatus[p_tail]);
		p_tail = (p_tail == P_MAX-1) ? 0 : ++p_tail;
	}

	return;
}


/*
 * Signal handler for SIGHUP
 */
void sigHUPHandler(int sig)
{
	sigHUP = TRUE;
	signal(SIGHUP, sigHUPHandler);
	return;
}
