/* wmtuxtime.c -- WindowMaker dockable applet portion of TuxTime
 *
 * Copyright (c) 1997-99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log:$
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

static char const rcsid[]="$Id:$";


#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#include<sys/stat.h>
#include<errno.h>
#include<signal.h>
#include<paths.h>
#include<pwd.h>
#include<features.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif
#include<X11/X.h>
#include<X11/xpm.h>


#include"wmtuxtime.h"
#include"wmgeneral.h"
#include"wmtuxtime.xpm"
#include"wmtuxtime_mask.xbm"
#include"sci.h"
#include"hci.h"
#include"alarms.h"

#define PID_FILE "wmtuxtime.pid"
static int modules;
static BatteryAlarm low,critical;
static int lowflag = 0;
static int criticalflag = 0;

#define MINUTES(h,m) ((h*60)+m)

enum {
	BATTERY_NORMAL   = 0x01,
	BATTERY_LOW      = 0x02,
	BATTERY_CRITICAL = 0x04
};

#define TIME_BATTERY_ALERT \
"You computer's battery power is low.\n\
You have approximately %d minutes\n\
left on this battery"

#define PERCENT_BATTERY_ALERT \
"You computer's battery power is low.\n\
Only %d%% of total battery capacity\n\
remains"

#define EMERGENCY_BATTERY_ALERT \
"You have almost no battery\n\
capacity left on this battery!\n\
Find an alternative power source\n\
immediately!"


int	CriticalLevel = 10;
int	LowLevel      = 40;
float   BlinkRate     = 3.0;	 /* blinks per second */
float   UpdateRate    = 0.8;     /* Number of updates per second */
int	Beep 	      = 0;	 /* Controls beeping when you get to CriticalLevel: Off by default */
int	Volume	      = 50;	 /* ring bell at 50% volume */
int	Alert	      = 0;  	 /* Controls whether alert is sent to all users via wall: Off by default  */
float  	LAlertRate     = 300.0;   /* send alert every 5 minutes when Low */
float  	CAlertRate     = 120.0;   /* send alert every 2 minutes when Critical */


/*
 * Process the command line arguments
 */
void ParseCommand(int argc, char *argv[])
{
	char *cmdline;
	int i,j;
	char puke[20];
   
	for (i=1;i<argc;i++) {
		cmdline = argv[i];

		if (cmdline[0]=='-') {
			switch(cmdline[1]) {
				case 'd': 
					i++;
					break;
				case 'A':
					Alert = 1;
					LAlertRate = atof(argv[++i]);
					CAlertRate = atof(argv[++i]);
					break;
				case 'b':
					BlinkRate = atof(argv[++i]);
					break;
				case 'C':
					CriticalLevel = atoi(argv[++i]);
					break;
				case 'L':
					LowLevel = atoi(argv[++i]);
					break;
				case 'B':
					Beep = 1;
					Volume = atoi(argv[++i]);
					break;
				default:
					printf("USAGE");
					exit(1);
			}
		}
	}

	return;
}


/*
 * Catch any signals and exit the program nicely
 */
void CatchSignal(int x)
{
	fprintf(stderr, "wmtuxtime: caught signal %d, cleaning up...\n", x);

	SciCloseInterface();
	unlink(_PATH_VARRUN PID_FILE);
	exit(0);
}


/*
 * Install signal handlers so we can clean up nicely on an error
 */
void HandleSignals(void)
{
	signal(SIGHUP, CatchSignal);
	signal(SIGINT, CatchSignal);
	signal(SIGQUIT, CatchSignal);
	signal(SIGILL, CatchSignal);
	signal(SIGTRAP, CatchSignal);
	signal(SIGABRT, CatchSignal);
	signal(SIGIOT, CatchSignal);
	signal(SIGFPE, CatchSignal);
	signal(SIGKILL, CatchSignal);
	signal(SIGSEGV, CatchSignal);
	signal(SIGPIPE, CatchSignal);
	signal(SIGTERM, CatchSignal);
	signal(SIGCHLD, CatchSignal);
	signal(SIGSTOP, CatchSignal);
	signal(SIGTTIN, CatchSignal);
	signal(SIGTTOU, CatchSignal);
	signal(SIGUSR1, SIG_IGN);

	return;
}


/*
 *  This routine handles button presses. Pressing the 'S' button
 *  invokes 'apm -S' to place the machine into standby mode. And 
 *  pressing the 'Z' buton invokes 'apm -s' to place the machine 
 *  into suspend mode.
 *
 *  Note: in order for users other than root to be able to run 
 *        'apm -s' and 'apm -S', you need to make apm suid (i.e. 
 *        run 'chmod +s /usr/bin/apm' as root).  This will allow
 *        'normal' users to execute apm with root privilages.
 *
 */
void pressEvent(XButtonEvent *event)
{	
	int x,y;
	
	x = event->x;
	y = event->y;

	if(x>=5 && y>=48 && x<=17 && y<=58) {
		/* next arrow button */

	 	copyXPMArea(5, 106, 13, 11, 5, 48);
		RedrawWindow();
		system("apm -S");
		usleep(2000000L);
	} else if (x>=46 && y>=48 && x<=58 && y<=58){
		/* alarm settings */

		copyXPMArea(21, 106, 13, 11, 46, 48);
		RedrawWindow();
		system("apm -s");
		usleep(2000000L);
	} else if (x>=46 && y>=48 && x<=58 && y<=58){
		/* suspend call */

		copyXPMArea(21, 106, 13, 11, 46, 48);
		RedrawWindow();
		system("apm -s");
		usleep(2000000L);
	} else if (x>=46 && y>=48 && x<=58 && y<=58){
		/* machine settings */

		copyXPMArea(21, 106, 13, 11, 46, 48);
		RedrawWindow();
		system("apm -s");
		usleep(2000000L);
	}


	return;
}


/*
 * Setup the alarm values by reading them in from the config file
 */
void SetupAlarms(char *config)
{
	char buffer[8];

	low.percent = GetConfigInt("AlarmLow", "Percent", 20, config);
	low.trigger = GetConfigInt("AlarmLow", "Trigger", 3, config);
	GetConfigString("AlarmLow", "Time", "00:20", buffer, 8, config);
	ConvertTime(buffer, &(low.hour), &(low.minute));
	critical.percent = GetConfigInt("AlarmCritical", "Percent", 10, config);
	critical.trigger = GetConfigInt("AlarmCritical", "Trigger", 3, config);
	GetConfigString("AlarmCritical", "Time", "00:10", buffer, 8, config);
	ConvertTime(buffer, &(critical.hour), &(critical.minute));

	return;
}


/*
 * Test to see if any battery alarm dialogs need displaying
 */
void AlarmTest(int type, int *flag, BatteryAlarm *settings, int lifepercent,
	int lifetime)
{
	*flag = 0;

	if ((settings->trigger==TIME_ALARM) || (settings->trigger==BOTH_ALARM))
		if ((lifetime<=MINUTES(settings->hour, settings->minute))
					&& (((*flag) & TIME_ALARM)==0)) {
			BatteryAlarmDialog(type | TIME_ALARM, lifetime);
			*flag |= TIME_ALARM;
		} else if (lifetime>MINUTES(settings->hour, settings->minute)) {
				*flag ^= TIME_ALARM;
		}
			
	if ((settings->trigger==PERCENT_ALARM) || (settings->trigger==BOTH_ALARM))
		if ((lifepercent<=settings->percent) &&
					(((*flag) & PERCENT_ALARM)==0)) {
			BatteryAlarmDialog(type | PERCENT_ALARM, lifepercent);
			*flag |= PERCENT_ALARM;
		} else if (lifepercent>settings->percent) {
			*flag ^= PERCENT_ALARM;
		}

	return;
}


/*
 * Update the battery life remaining display, and switch settings if the
 * power source has changed
 */
void Update(void)
{
	SciRegisters reg;
	int timelife,percentlife;
	int hour,minute,state;
	char buffer[8];

	timelife = 0;
	percentlife = 0;

	/* get the percent and time remaining */

	SciOpenInterface();
	reg.setting = SCI_BATTERY_PERCENT;
	if (SciGet(&reg)==SCI_SUCCESS)
		percentlife = (100*reg.current)/reg.possible;
	else
		precentlife = -1;

	reg.setting = SCI_BATTERY_TIME;
	if (SciGet(&reg)==SCI_SUCCESS)
		timelife = reg.current;
	else
		timelife = -1;

	/* test for emergency battery alarm */

	if (percentlife<3)
		fprintf(stderr,"EMERGENGY BATTERY ALARM\n");

	/* calculate the user settable battery conditions/alarms */

	hour = SCI_HOUR(timelife);
	minute = SCI_MINUTE(timelife);
	AlarmTest(LOW_ALARM, &flag, &low, percentlife, MINUTES(hour, minute));
	if (flag!=0)

	AlarmTest(CRITICAL_ALARM, &flag, &critical, percentlife, MINUTES(hour, minute));
		
	return;
}


/*
 * the entry point of wmTuxTime
 */
int main(int argc, char *argv[])
{
	int time_left,hour_left,min_left,digit;
	XEvent event;
	int m,mMax,n,nMax,k,Toggle;
	long int r, rMax, s, sMax;
	FILE *str,*fp;

	/* get the necessary I/O permissions */

	if (ioperm(0xb2, 1, 1)) {
		fprintf(stderr, "tuxtime: can't get I/O permissions.\n");
		return 1;
	}

	SciCloseInterface();

	/* do some quick checks on the laptop */

	flag = SciSupportCheck(&version);
	if (flag==1) {
		fprintf(stderr, "wmtuxtime: this computer is not supported\n");
		return 1;
	}

	bios = HciGetBiosVersion();
	if (bios==0) {
		fprintf(stderr, "wmtuxtime: unable to get BIOS version\n");
		return 1;
	}

	if (HciGetMachineID(&id)==HCI_FAILURE) {
		fprintf(stderr, "wmtuxtime: unable to get machine identification\n");
		return 1;
	}

	/* check to see if a copy of wmTuxTime is already running */

	if (!access(_PATH_VARRUN PID_FILE, R_OK)) {
		if ((str = fopen(_PATH_VARRUN PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other TuxTime died unexpectedly */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "tuxtime: already running as "
					"process %d.\n", pid);
				exit(1);
			}

			fprintf(stderr, "tuxtime: process %d appears to have "
				"died, continuing\n", pid);
			unlink(_PATH_VARRUN PID_FILE);
		}
	}

	/* get the path of the tuxtimerc file */

	pw = getpwuid(getuid());
	config = (char *) malloc(strlen(pw->pw_dir)+12);
	if (config==NULL) {
		fprintf(stderr, "tuxtime: unable to allocate sufficent memory, "
			"exiting\n");
		exit(1);
		}
	strcpy(config, pw->pw_dir);
	strcat(config, "/.tuxtimerc");

	/* test to see if tuxtimerc file exists and if no create one */

	if ((stat(config, &info)!=0) && (errno==ENOENT)) {
		CreateDefaultRC(config);
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(_PATH_VARRUN PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}


	/* install signal handling after GTK toolkit is initialize otherwise
	   it will override our signal handling */

	HandleSignals();


	BlinkRate = 3.0;
	UpdateRate = 1.0/1.25;

	/* parse any command line arguments. */

	ParseCommand(argc, argv);

	BlinkRate = (BlinkRate >= 0.0) ? BlinkRate : -1.0*BlinkRate;
	UpdateRate = (UpdateRate >= 0.0) ? UpdateRate : -1.0*UpdateRate;
   

    nMax = (int)( 1.0e6/(2.0*UpdateRate*DELAY)  );
    mMax = (BlinkRate > 0.0) ? (int)( 1.0e6/(2.0*BlinkRate*DELAY)  ) : nMax;
    rMax = (int)( LAlertRate*1.0e6/(2.0*DELAY)  );
    sMax = (int)( CAlertRate*1.0e6/(2.0*DELAY)  );


	/* open the X11 windows */

	openXwindow(argc, argv, wmapm_master, wmapm_mask_bits,
		wmapm_mask_width, wmapm_mask_height);

	/* loop until we die... */

	n = 32000;
	m = 32000;
	r = rMax+1;
	s = sMax+1;
	for (;;) {


	/*
	 *  Only process apm info only every nMax cycles of this
	 *  loop. We run it faster to catch the xevents like button 
	 *  presses and expose events, etc...
	 *  
	 *  DELAY is set at 0.00625 seconds, so process apm info
	 *  every 1.25 seconds...
	 *
	 */
	if (n>nMax){
		n = 0;

		/* check AC status. */

		if (SciACPower()==SCI_MAINS)
			copyXPMArea(68, 6, 26, 7, 31, 35);
		else
			copyXPMArea(68, 20, 26, 7, 31, 35);

		/* paste up the default charge status and time */

		copyXPMArea(104,  6,  5, 7, 6, 7);
		copyXPMArea(83, 93, 41, 9, 15, 7);

		/* check to see if we are charging */

		if ( (int)(my_cur_info.battery_status) == 3){
			copyXPMArea(98,  6, 5, 7,  6,  7);
			copyXPMArea(75, 81, 1, 2, 17,  9);
			copyXPMArea(75, 81, 1, 2, 17, 12);
		}  

	    /* repaint buttons. */
	    copyXPMArea(42, 106, 13, 11, 5, 48);
	    copyXPMArea(57, 106, 13, 11, 46, 48);

	    /* 
	     *  Paste up the "Time Left". This time means:
	     *  
	     *         If not charging: Time left before battery drains to 0% 
	     *         If charging:     Time left before battery gets to maximum
	     *
	     */

	    if (my_cur_info.battery_time >= 1440) {

		/*
		 *  If battery_time is too large, it likely means that there is
		 *  no charging or discharging going on. So just display a "null"
		 *  indicator (--:--). 
		 *
		 */
		copyXPMArea(83, 106, 41, 9, 15, 7);

	    } else if (my_cur_info.battery_time >= 0) {

		time_left = (my_cur_info.using_minutes) ? my_cur_info.battery_time : my_cur_info.battery_time / 60;

	        hour_left = time_left / 60;
	        min_left  = time_left % 60;

	        copyXPMArea( (hour_left / 10) * 7 + 5, 93, 7, 9, 21, 7); 	/* Show 10's (hour) */
	        copyXPMArea((hour_left % 10) * 7 + 5, 93, 7, 9, 29, 7); 	/* Show 1's (hour)  */
	        copyXPMArea(76, 93, 2, 9, 38, 7);		 		/* colon  	    */
	        copyXPMArea((min_left / 10) * 7 + 5, 93, 7, 9, 42, 7); 		/* Show 10's (min)  */
	        copyXPMArea((min_left % 10) * 7 + 5, 93, 7, 9, 50, 7); 		/* Show 1's (min)   */

	    }
	
	    /* 
	     *   Do Battery Percentage.
 	     */
	    copyXPMArea(76, 81, 19, 7, 7, 34);            		/* Show Default % */
	    copyXPMArea(66, 31, 49, 9, 7, 21);           		/* Show Default Meter */
	    if (my_cur_info.battery_percentage == 100){
	            copyXPMArea(15, 81, 1, 7,  7, 34);             	/* If 100%, show 100% */
	            copyXPMArea( 5, 81, 6, 7,  9, 34);
	            copyXPMArea( 5, 81, 6, 7, 15, 34);
	            copyXPMArea(64, 81, 7, 7, 21, 34);            	/* Show '%' */
	    	    copyXPMArea(66, 42, 49, 9, 7, 21);           	/* Show Meter */
	    } else {

		if (my_cur_info.battery_percentage >= 10)
		    copyXPMArea((my_cur_info.battery_percentage / 10) * 6 + 4, 81, 6, 7,  9, 34);  	/* Show 10's */
		copyXPMArea((my_cur_info.battery_percentage % 10) * 6 + 4, 81, 6, 7, 15, 34);		/* Show 1's */
		copyXPMArea(64, 81, 7, 7, 21, 34);							/* Show '%' */

		/*
		 *  Show Meter
		 */
		k = my_cur_info.battery_percentage * 49 / 100;
		copyXPMArea(66, 42, k, 9, 7, 21);
		if (k%2) 
		    copyXPMArea(66+k-1, 52, 1, 9, 7+k-1, 21);
		else     
		    copyXPMArea(66+k, 52, 1, 9, 7+k, 21);
	    }



	} else {

	    /*
	     *  Update the counter. When it hits nMax, we will 
	     *  process /proc/apm information again.
	     */
	    n++;

	}



	/*
	 *  This controls Critical Alerts
  	 */
	if (Alert){
	    if (( (int)(my_cur_info.battery_status) == 2)
		    ||( (int)(my_cur_info.battery_percentage) <= CriticalLevel )){

	        if (s>sMax){

	            s = 0;
	            fp = popen("wall", "w");
	            fprintf(fp, "Battery is critical!. Percent: %d\n", (int)(my_cur_info.battery_percentage));
	            pclose(fp);

	        } else {

	            /*
	             *  Update the counter. 
	             */
	            ++s;

	        }

	    } else if (( (int)(my_cur_info.battery_status) == 1)
		    ||( (int)(my_cur_info.battery_percentage) <= LowLevel )){

	        if (r>rMax){

	            r = 0;
	            fp = popen("wall", "w");
	            fprintf(fp, "Battery is low. Percent: %d\n", (int)(my_cur_info.battery_percentage));
	            pclose(fp);

	        } else {

	            /*
	             *  Update the counter. 
	             */
	            ++r;

	        }

	    }
	}



		/* Process any pending X events. */

		while (XPending(display)) {
			XNextEvent(display, &event);
			switch(event.type){
				case Expose:
					RedrawWindow();
					break;
				case ButtonPress:
					pressEvent(&event.xbutton);
					break;
				case ButtonRelease:
                        		break;
			}
		}

		/* Redraw and wait for next update  */

		RedrawWindow();
		usleep(DELAY);
	}

	return 0;
}


