/* fan.c -- turn the internal fan on/off in a Toshiba Pentium(tm) laptop.
 *
 * Copyright (c) 1996,97,98  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: fan.c,v $
 * Revision 3.3  1998/12/30 23:42:45  jab
 * added toggle option to turn the fan off if on and vice versa
 * started move to Hardware Configuration Interface routines
 *
 * Revision 3.2  1998/09/11 22:05:23  jab
 * switched to using routines from the System Configuration Interface
 * added GNU style long options
 * added auto option which turns the fan off if using battery and on if mains
 *
 * Revision 3.1  1998/08/22 10:33:27  jab
 * minor tidy ups to the code
 *
 * Revision 3.0  1998/07/11 14:01:38  jab
 * New selective switching method based on the return from GetMachineID,
 * should work on all known models
 *
 * Revision 2.2  1998/05/30 22:50:03  jab
 * hopefully fixed problems with reporting of fan status
 *
 * Revision 2.1  1998/05/08 22:52:17  jab
 * now drop root priveleges as soon as permision on the ports granted
 *
 * Revision 2.0  1998/01/31 16:00:23  jab
 * Changed to new method of turning the fan on/off, which should
 * work on a wider range of Toshiba laptops
 *
 * Revision 1.5  1997/05/23 13:17:25  jab
 * change the command line option processing to only deal with the first
 *
 * Revision 1.4  1997/04/29 21:26:11  jab
 * changed the name of the port variables to reflect their real status
 *
 * Revision 1.3  1996/08/01 14:25:36  jab
 * Added logging of changes in fan status via syslogd(8)
 *
 * Revision 1.2  1996/07/30 18:11:16  jab
 * fixed reporting of unknown command line options
 *
 * Revision 1.1  1996/06/25 21:47:42  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

static char const rcsid[]="$Id: fan.c,v 3.3 1998/12/30 23:42:45 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<unistd.h>
#include<syslog.h>
#include<signal.h>
#include<paths.h>
#include<pwd.h>
#include<features.h>
#include<sys/stat.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif

#include"hci.h"
#include"sci.h"

enum { DEFAULT, PORTAGE, TECRA };
enum { ON, OFF, STATUS, AUTO, TOGGLE };

#define PID_FILE _PATH_VARRUN "fan.pid"

int method;
char *name;

int HciFanOn(int method)
{
	unsigned char status;

	switch (method) {
		case PORTAGE:
			asm("cli\n\t" \
				"movb $0xbe,%al\n\t" \
				"outb %al,$0xe4\n\t" \
				"inb $0xe5,%al\n\t" \
				"andb $0xfe,%al\n\t" \
				"movb %al,%ah\n\t" \
				"movb $0xbe,%al\n\t" \
				"outb %al,$0xe4\n\t" \
				"movb %ah,%al\n\t" \
				"outb %al,$0xe5\n\t" \
				"sti\n");
			status = 0x00;
			break;
		case TECRA:
			asm("cli\n\t" \
				"movw $0x00e4,%dx\n\t" \
				"movb $0xe0,%al\n\t" \
				"outb %al,%dx\n\t" \
				"incw %dx\n\t" \
				"inb %dx,%al\n\t" \
				"orw $0x0001,%ax\n\t" \
				"decw %dx\n\t" \
				"movb %al,%ah\n\t" \
				"movb $0xe0,%al\n\t" \
				"outw %ax,%dx\n\t" \
				"sti\n");
			status = 0x00;
			break;
		default:
			asm("movw $0xffff,%%ax\n\t" \
				"movw $0x0004,%%bx\n\t" \
				"movw $0x0001,%%cx\n\t" \
				"inb $0xb2,%%al\n\t" \
				"movb %%ah,%0\n" \
				:"=m" (status) : : "memory" );
			break;
	}

	return (int) status;
}

int HciFanOff(int method)
{
	unsigned char status;

	switch (method) {
		case PORTAGE:
			asm("cli\n\t" \
				"movb $0xbe,%al\n\t" \
				"outb %al,$0xe4\n\t" \
				"inb $0xe5,%al\n\t" \
				"orb $0x01,%al\n\t" \
				"movb %al,%ah\n\t" \
				"movb $0xbe,%al\n\t" \
				"outb %al,$0xe4\n\t" \
				"movb %ah,%al\n\t" \
				"outb %al,$0xe5\n\t" \
				"sti\n");
			status = 0x00;
			break;
		case TECRA:
			asm("cli\n\t" \
				"movw $0x00e4,%dx\n\t" \
				"movb $0xe0,%al\n\t" \
				"outb %al,%dx\n\t" \
				"incw %dx\n\t" \
				"inb %dx,%al\n\t" \
				"andw $0xfffe,%ax\n\t" \
				"decw %dx\n\t" \
				"movb %al,%ah\n\t" \
				"movb $0xe0,%al\n\t" \
				"outw %ax,%dx\n\t" \
				"sti\n");
			status = 0x00;
			break;
		default:
			asm("movw $0xffff,%%ax\n\t" \
				"movw $0x0004,%%bx\n\t" \
				"movw $0x0000,%%cx\n\t" \
				"inb $0xb2,%%al\n\t" \
				"movb %%ah,%0\n" \
				:"=m" (status) : : "memory" );
			break;
	}

	return (int) status;
}

int HciFanStatus(int method)
{
	int status;
	unsigned char ah;
	unsigned short ax,cx;

	switch(method) {
		case PORTAGE:
			asm("cli\n\t" \
				"movb $0xbe,%%al\n\t" \
				"outb %%al,$0xe4\n\t" \
				"inb $0xe5,%%al\n\t" \
				"andw $0x0001,%%ax\n\t" \
				"movw %%ax,%0\n\t" \
				"sti\n" \
 				:"=m" (ax) : : "memory" );
			status = (ax==0x00) ? 0x01:0x00;
			break;
		case TECRA:
			asm("cli\n\t" \
				"movw $0x00e4,%%dx\n\t" \
				"movb $0xe0,%%al\n\t" \
				"outb %%al,%%dx\n\t" \
				"incw %%dx\n\t" \
				"inb %%dx,%%al\n\t" \
				"andw $0x0001,%%ax\n\t"
				"movw %%ax,%0\n\t" \
				"sti\n" \
				:"=m" (ax) : : "memory" );
			status = (int) ax;
			break;
		default:
			asm("movw $0xfefe,%%ax\n\t" \
				"movw $0x0004,%%bx\n\t" \
				"movw $0x0000,%%cx\n\t" \
				"inb $0xb2,%%al\n\t" \
				"movw %%cx,%0\n\t" \
				"movb %%ah,%1\n" \
 				:"=m" (cx), "=m" (ah) : : "memory" );
			status = (ah==0x00) ? (int) cx & 0x0001 : -1*(int) ah;
			break;
	}

	/* if I remove the next line I get seg faults under gcc 2.7.2.3 !!! */

	printf("");

	return status;
}

void FanOn(char *namet)
{
	HciFanOn(method);
	syslog(LOG_INFO, "cooling fan turned on by %s", name);
	return;
}

void FanOff(char *namet)
{
	HciFanOff(method);
	syslog(LOG_INFO, "cooling fan turned off by %s", name);
	return;
}


static void Catch(int sig)
{
	syslog(LOG_INFO, "Exiting" );
	unlink(PID_FILE);
	exit(0);
}


/*
 * Test to see if fan is already running in daemon mode
 */
void Running(void)
{
	int pid;
	FILE *str;

	/* check to see if fan daemon is already running */

	if (!access(PID_FILE, R_OK)) {
		if ((str = fopen(PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other daemon died unexpectedly */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "fan: daemon already running "
					"as process %d.\n", pid);
				exit(1);
			}

			fprintf(stderr, "fan: process %d appears to have "
				"died, continuing\n", pid);
			unlink(PID_FILE);
		}
	}
} 

/*
 * Enter daemon mode turning the fan on constantly while the mains is connected
 */
void Daemon(void)
{
	int pid;
	FILE *str;

	/* register some signal handlers */

	if (signal(SIGINT, SIG_IGN)!=SIG_IGN) signal(SIGINT, Catch);
	if (signal(SIGQUIT, SIG_IGN )!=SIG_IGN) signal(SIGQUIT, Catch);
	if (signal(SIGTERM, SIG_IGN)!=SIG_IGN) signal(SIGTERM, Catch);
	signal(SIGUSR1, SIG_IGN);

	/* parent */

	if ((pid=fork())) {
		if ((str=fopen(PID_FILE, "w"))) {
			fprintf(str, "%d\n", pid);
			fclose(str);
		}
		exit(0);
	}

	/* child */

	if (pid < 0) {
		syslog(LOG_INFO, "fork() failed: %m");
		unlink(PID_FILE);
		exit(1);
	} else {
		syslog(LOG_INFO, "entering daemon mode");
	}

	/* child - Follow the daemon rules in W. Richard Stevens.Advanced
	   Programming in the UNIX Environment (Addison-Wesley Publishing Co.,
	   1992). Page 417.). */

	if (setsid()<0) {
		syslog(LOG_INFO, "setsid() failed: %m");
		unlink(PID_FILE);
		exit(1);
	}

	chdir("/");
	umask(0);

	/* enter into an infinite loop */

	for (;;) {
		if (SciACPower()==SCI_MAINS) HciFanOn(method);
		sleep(1);
	}

}


int main(int argc, char *argv[])
{
	int id;
	struct passwd *pw;
	char *namet;

	/* get the necessary I/O permissions */

	if (iopl(3)) {
		fprintf(stderr, "fan: can't get I/O permissions.\n");
		return 1;
	}

	/* drop root priveleges to reduce security risk of running suid root */
/*
	seteuid(getuid());
	setegid(getgid());
*/
	/* check to see if already running in daemon mode */

	Running();

	/* select a method based on the model we are running on */

	method = DEFAULT;
	if (HciGetMachineID(&id)==SCI_SUCCESS) {
		if (id==0xfccb)
			method = PORTAGE;
		else if (id==0xfccc)
			method = TECRA;
	}

	/* check to make sure laptop has a fan */

	if (HciFanStatus(method)<0x00) {
		fprintf(stderr, "fan: laptop does not have cooling fan.\n");
		return 1;
	}

	/* open connection to system logger */

	openlog("fan", LOG_PID | LOG_CONS, LOG_USER);

	/* get user name */

	pw = getpwuid(getuid());
	name = pw ? pw->pw_name : getenv("LOGNAME");

	/* process command line arguments */

	if (--argc>0) {
		argv++;
		if (!strcmp(argv[0], "-n") || !strcmp(argv[0], "--on")) {
			FanOn(name);
		} else if (!strcmp(argv[0], "-f") || !strcmp(argv[0], "--off")) {
			FanOff(name);
		} else if (!strcmp(argv[0], "-a") || !strcmp(argv[0], "--auto")) {
			if (SciACPower()==SCI_BATTERY) {
				FanOff(name);
			} else {
				FanOn(name);
			}
		} else if (!strcmp(argv[0], "-t") || !strcmp(argv[0], "--toggle")) {
			if (HciFanStatus(method)==0x00)
				FanOn(name);
			else
				FanOff(name);
		} else if (!strcmp(argv[0], "-d") || !strcmp(argv[0], "--daemon")) {
			Daemon();
		} else {
			fprintf(stderr, "fan: illegal option %s\n"
				"Usage: fan [-n|f|a|t|d]\n", argv[0]);
			return 1;
		}
	}

	/* Toshiba's fan.exe waits here for 0.1 seconds, so we do the same */
	
	usleep(100000);

	/* print the current status of the fan */

	if (HciFanStatus(method)==0x00)
		printf("Fan is off.\n");
	else
		printf("Fan is on.\n");

	return 0;
}
