/* dispswitch.c -- utility to change the video output device
 *
 * Copyright (c) 1999  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: dispswitch.c,v $
 * Revision 1.1  1999/08/09 20:15:26  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

static const char rcsid[]="$Id: dispswitch.c,v 1.1 1999/08/09 20:15:26 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#include<signal.h>
#include<paths.h>
#include<pwd.h>
#include<features.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif
#include<gtk/gtk.h>

#include"sci.h"
#include"hci.h"

#include"displayswitch.xpm"

#define PID_FILE "dispswitch.pid"

#undef VERSION
#define VERSION "\
Display Switch  version 1.0\n\
Copyright (c) 1999 Jonathan A. Buzzard <jonathan@buzzard.org.uk>\n"

#define USAGE \
"Usage: dispswitch [OPTION]\n\n\
  -i,--internal      select the LCD display\n\
  -e,--external      select the external display\n\
  -s,--simultaneous  use both the LCD and external display\n\
  -h,--help          display this help message\n\
  -v,--version       display version\n\
Report bugs to jonathan@buzzard.org.uk\n"


GtkWidget *internal,*external,*simultaneous;
unsigned char output;


struct Options {
	char *shortopt;
	char *longopt;
	unsigned short mode;
};

struct Options opts[] = {
	{"-i", "--internal", HCI_INTERNAL},
	{"-e", "--external", HCI_EXTERNAL},
	{"-s", "--simultaneous", HCI_SIMULTANEOUS},
	{NULL, NULL, 0}
};


/*
 * if we recieve a signal, exit nicely freeing up resources
 */
void CatchSignal(int x)
{
	fprintf(stderr, "dispswitch: caught signal %d, cleaning up...\n", x);

	unlink(_PATH_VARRUN PID_FILE);
	gtk_main_quit();
}


/*
 * install signal handlers so we can exit nicely it necessary
 */
void HandleSignals(void)
{
	signal(SIGHUP, CatchSignal);
	signal(SIGINT, CatchSignal);
	signal(SIGQUIT, CatchSignal);
	signal(SIGILL, CatchSignal);
	signal(SIGTRAP, CatchSignal);
	signal(SIGABRT, CatchSignal);
	signal(SIGIOT, CatchSignal);
	signal(SIGFPE, CatchSignal);
	signal(SIGKILL, CatchSignal);
	signal(SIGSEGV, CatchSignal);
	signal(SIGPIPE, CatchSignal);
	signal(SIGTERM, CatchSignal);
	signal(SIGCHLD, CatchSignal);
	signal(SIGSTOP, CatchSignal);
	signal(SIGTTIN, CatchSignal);
	signal(SIGTTOU, CatchSignal);
	signal(SIGUSR1, SIG_IGN);

	return;
}


/*
 * deal with window manager delete events
 */
void Deleted(GtkWidget *widget, gpointer *data)
{
	unlink(_PATH_VARRUN PID_FILE);
	gtk_main_quit();
}


/*
 * Change the video out device
 */
void SetDisplayCallback(GtkWidget *widget, gpointer *data)
{
	HciRegisters reg;

	reg.ax = HCI_SET;
	reg.bx = HCI_VIDEO_OUT;
	reg.cx = (int) data;
	HciFunction(&reg);
	output = (int) data;

	return;
}


/*
 * update setting in case it is changed with a hotkey combination
 */
gint Update(gpointer data)
{
	HciRegisters reg;

	reg.ax = HCI_GET;
	reg.bx = HCI_VIDEO_OUT;
	HciFunction(&reg);

	if (output==(int) reg.cx)
		return TRUE;

	switch (reg.cx) {
		case HCI_INTERNAL:
			gtk_toggle_button_set_state(
				GTK_TOGGLE_BUTTON(internal), TRUE);
		case HCI_EXTERNAL:
			gtk_toggle_button_set_state(
				GTK_TOGGLE_BUTTON(external), TRUE);
		case HCI_SIMULTANEOUS:
			gtk_toggle_button_set_state(
				GTK_TOGGLE_BUTTON(simultaneous), TRUE);
	}

	output = (int) reg.cx;

	return TRUE;
}

/*
 * process the non-GTK command line arguments
 */
void ProcessComandLine(int *argc, char ***argv)
{
	int i,j,quit;

	quit = 0;
	for (i=1;i<*argc;i++) {
		if ((!strcmp((*argv)[i], "-h")) || (!strcmp((*argv)[i], "--help"))) {
			printf(USAGE);
			unlink(_PATH_VARRUN PID_FILE);
			exit(0);
		}
		if ((!strcmp((*argv)[i], "-v")) || (!strcmp((*argv)[i], "--version"))) {
			printf(VERSION);
			unlink(_PATH_VARRUN PID_FILE);
			exit(0);
		}

		/* step through our command line argument structure */

		for (j=0;opts[j].shortopt;j++) {
			if ((!strcmp(opts[j].shortopt, (*argv)[i])) ||
				(!strcmp(opts[j].longopt, (*argv)[i]))) {
					quit = 1;
					SetDisplayCallback(NULL, (gpointer)
						((int) opts[j].mode));
					break;
				}
		}
		if (!opts[j].shortopt) {
			fprintf(stderr, "dispswitch: unrecognised option %s\n",
				(*argv)[i]);
			unlink(_PATH_VARRUN PID_FILE);
			exit(1);
		}

	}

	/* exit if requested */

	if (quit>0) {
		unlink(_PATH_VARRUN PID_FILE);
		exit(0);
	}

	return;
}


/*
 * the entry point of Display Switch
 */
int main(int argc, char *argv[])
{
	int pid,version;
	HciRegisters reg;
	SciRegisters r;
	FILE *str;
	GtkWidget *window,*box,*row,*frame,*button,*label;
	GtkStyle *style;
	GdkPixmap *icon;
	GdkBitmap *mask;
	GSList *radio;


	/* get the necessary I/O permissions */

	if (ioperm(0xb2, 1, 1)) {
		fprintf(stderr, "dispswitch: can't get I/O permissions.\n");
		return 1;
	}

	/* do some quick checks on the laptop */

	if (SciSupportCheck(&version)==1) {
		fprintf(stderr, "dispswitch: this computer is not supported\n");
		return 1;
	}

	r.setting = 0x8300;
	r.current = 0;
	r.possible = 0;
	SciCloseInterface();
	SciOpenInterface();
	SciGet(&r);
	SciCloseInterface();
	printf("0x%04x  0x%04x  0x%04x\n", r.setting, r.current, r.possible);

	reg.ax = HCI_GET;
	reg.bx = HCI_VIDEO_OUT;
	if (HciFunction(&reg)==HCI_NOT_SUPPORTED) {
		fprintf(stderr, "dispswitch: this computer is not supported\n");
		return 1;
	}

	/* check to see if a copy of Display Switch is already running */

	if (!access(_PATH_VARRUN PID_FILE, R_OK)) {
		if ((str = fopen(_PATH_VARRUN PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other dispswitch died */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "dispswitch: already running as"
					" process %d.\n", pid);
				exit(1);
			}


			fprintf(stderr, "dispswitch: process %d appears to have"
				" died, continuing\n", pid);
			unlink(_PATH_VARRUN PID_FILE);
		}
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(_PATH_VARRUN PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}

	/* initialize the GTK toolkit */

	gtk_init(&argc, &argv);

	/* install signal handling after GTK toolkit is initialize otherwise
	   it will override our signal handling */

	HandleSignals();

	/* Gtk options are cleared, process remaining command line arguments */

	ProcessComandLine(&argc, &argv);

	/* create the top level window */

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_container_border_width(GTK_CONTAINER(window), 1);
	gtk_signal_connect(GTK_OBJECT(window), "delete_event",
		GTK_SIGNAL_FUNC(Deleted), NULL);

	/* add window tile and icon */

	gtk_window_set_title(GTK_WINDOW(window), "Display Switch");
	style = gtk_widget_get_style(window);
	gtk_widget_show(window);
	icon = gdk_pixmap_create_from_xpm_d(window->window, &mask,
		&style->bg[GTK_STATE_NORMAL], (gchar **)displayswitch_xpm);
	gdk_window_set_icon(window->window, NULL, icon, mask);

	box = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(window), box);

	frame = gtk_frame_new("Display Switch");
/*	gtk_container_border_width(GTK_CONTAINER(frame), 16);*/
	gtk_box_pack_start(GTK_BOX(box), frame, TRUE, TRUE, 0);

	/* add close button */

	row = gtk_hbox_new(TRUE, 5);
	gtk_container_border_width(GTK_CONTAINER(row), 10);
	gtk_box_pack_start(GTK_BOX(box), row, FALSE, TRUE, 0);
	button = gtk_button_new_with_label("Close");
	gtk_widget_set_usize(button, 60, 20);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(Deleted), GTK_OBJECT(window));
	gtk_box_pack_start(GTK_BOX(row), button, FALSE, FALSE, 0);

	/* add radio buttons */

	box = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), box);

	label = gtk_label_new("Please select the display device.");
	gtk_misc_set_padding(GTK_MISC(label), 10, 10);
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, TRUE, 0);

	internal = gtk_radio_button_new_with_label(NULL, "Internal Display");
	gtk_box_pack_start(GTK_BOX(box), internal, FALSE, TRUE, 0);
	radio = gtk_radio_button_group(GTK_RADIO_BUTTON(internal));
	gtk_signal_connect(GTK_OBJECT(internal), "pressed",
		GTK_SIGNAL_FUNC(SetDisplayCallback), (gpointer) HCI_INTERNAL);

	external = gtk_radio_button_new_with_label(radio, "External Display");
	gtk_box_pack_start(GTK_BOX(box), external, FALSE, TRUE, 0);
	radio = gtk_radio_button_group(GTK_RADIO_BUTTON(external));
	gtk_signal_connect(GTK_OBJECT(external), "pressed",
		GTK_SIGNAL_FUNC(SetDisplayCallback), (gpointer) HCI_EXTERNAL);

	simultaneous = gtk_radio_button_new_with_label(radio,
		"Simultaneous Display");
	gtk_box_pack_start(GTK_BOX(box), simultaneous, FALSE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(simultaneous), "pressed",
		GTK_SIGNAL_FUNC(SetDisplayCallback),
		(gpointer) HCI_SIMULTANEOUS);

	/* force an update on the displayed output device */

	Update(NULL);

	gtk_widget_show_all(window);

	/* add the timeout for the update function */

	gtk_timeout_add(1000, Update, (gpointer) 1);

	/* start the program */

	gtk_main();

	return 0;
}
