/* alarm.c -- set the alarm power on time of a Toshiba laptop.
 *
 * Copyright (c) 1998-99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log:$
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

static char const rcsid[]="$Id:$";


#include<stdio.h>
#include<stdlib.h>
#include<unistd.h>
#include<syslog.h>
#include<signal.h>
#include<paths.h>
#include<time.h>
#include<pwd.h>
#include<features.h>
#include<sys/stat.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif

#include"sci.h"
#include"parsetime.h"


#undef VERSION
#define VERSION "\
Alarm version 1.0\n\
Copyright (c) 1998-99 Jonathan A. Buzzard <jonathan@buzzard.org.uk>\n"

#define USAGE \
"Usage: alarm [OPTION]\n\
Set the alarm power on time or date of a Toshiba laptop\n\n\
  -t,--time {time}   set the alarm power on time\n\
  -d,--date          set the alarm power on date\n\
  -h,--help          display this help message\n\
  -v,--version       display version\n\
Report bugs to jonathan@buzzard.org.uk\n"


int support;

enum {
	ALARM_TIME  = 0x0001,
	ALARM_DATE  = 0x0002,
	ALARM_EVERY = 0x0004,
	ALARM_YEAR  = 0x0008
};


int main(int argc, char *argv[])
{
	SciRegisters reg;
	time_t timer;
	int i,version,local;
	int hour,minute;
	struct tm *alarm;

	/* get the necessary I/O permissions */

	if (iopl(3)) {
		fprintf(stderr, "alarm: can't get I/O permissions.\n");
		return 1;
	}

	/* drop root priveleges to reduce security risk of running suid root */

	seteuid(getuid());
	setegid(getgid());

	/* do some quick checks on the laptop */

	if (SciSupportCheck(&version)==1) {
		fprintf(stderr, "alarm: this computer is not supported\n");
		return 1;
	}

	/* check the alarm features of the laptop */

	support = 0x0000;
	SciOpenInterface();

	reg.setting = SCI_ALARM_TIME;
	if (SciGet(&reg)==SCI_SUCCESS) {
		support &= ALARM_TIME;
	}

#define SCI_ALARM_EVERYDAY 0x0001

	reg.setting = SCI_ALARM_DATE;
	if (SciGet(&reg)==SCI_SUCCESS) {
		support &= ALARM_DATE;
		if ((reg.possible & SCI_ALARM_EVERYDAY)==0x0001)
			support &= ALARM_EVERY;
		if (SCI_YEAR(reg.possible)!=1990)
			support &= ALARM_YEAR;
	}

	SciCloseInterface();

	/* process command line arguments */

	local = 0;
	for (i=1;i<argc;i++) {
		if ((!strcmp(argv[i], "-h")) || (!strcmp(argv[i], "--help"))) {
			printf(USAGE);
			exit(0);
		}
		if ((!strcmp(argv[i], "-v")) || (!strcmp(argv[i], "--version"))) {
			printf(VERSION);
			exit(0);
		}
		if ((!strcmp(argv[i], "-l")) || (!strcmp(argv[i], "--localtime"))) {
			local = 1;
		}		
	}

	if (local==1)
		argv++;

	if (argc>optind) {
		timer = parsetime(i-optind, argv+optind);
	} else {
		timer = 0;
	}

	/* exit if the time was garbled or is an unrecognised option */

	if ((timer==0) && (argc>optind)) {
		fprintf(stderr, "alarm: garbled time or unrecognised option.\n");
		exit(1);
	}

	/* convert the calander time into UTC or localtime as necessary */

	if (local==1)
		alarm = localtime(&timer);
	else
		alarm = gmtime(&timer);

	printf("ALARM TIME STRING = %s", asctime(alarm));
	printf("ALARM TIME %02d:%02d\n", alarm->tm_hour, alarm->tm_min);
	printf("TIMER = %d\n",(int) timer);
/*	printf("ALARM TIME 0x%04x\n", SCI_TIME(alarm->tm_hour, alarm->tm_min));
*/
	hour = alarm->tm_hour;
	minute = alarm->tm_min;
	printf("ALARM TIME %02d:%02d\n", hour, minute);


	/* set the alarm time */

	if (((support&ALARM_TIME)==0) && (timer!=0)) {
		SciOpenInterface();
		reg.setting = SCI_ALARM_TIME;
		reg.current = SCI_ALARM_ENABLED | SCI_TIME(alarm->tm_hour,
			alarm->tm_min);
		SciSet(&reg);
		SciCloseInterface();
	}

	/* display the current alarm power on setting */

	SciOpenInterface();
	reg.setting = SCI_ALARM_TIME;
	if (SciGet(&reg)==SCI_SUCCESS) {
		if (SCI_TIME_ON(reg.current)==SCI_ALARM_ENABLED)
			printf("alarm power on set to %02d:%02d\n",
				SCI_HOUR(reg.current), SCI_MINUTE(reg.current));
		else
			printf("alarm power on disabled\n");
	} else {
		fprintf(stderr, "alarm: power on time not supported on "
			"this model.\n");
	};
	
	reg.setting = SCI_ALARM_DATE;
	if (SciGet(&reg)==SCI_SUCCESS) {
		if ((reg.possible & 0x0001)==0x0001) {
			printf("DATE: everyday\n");
		} else {
			printf("ax=0x%04x  bx=0x%04x  cx=0x%04x  dx=0x%04x\n",
				reg.attribute, reg.setting, reg.current,
				reg.possible);
		}
	}

	
	printf("year=%d month=%d day=%d\n", SCI_YEAR(reg.current),
		SCI_MONTH(reg.current), SCI_DAY(reg.current));
	
	
	
	SciCloseInterface();


	return 0;
}
