/* pe1.c -- Linux driver for the Xircom PE1 parallel port Ethenet adaptor 
 *
 * Copyright (c) 1996-97  Jonathan A. Buzzard (jab@hex.prestel.co.uk)
 *
 * Valuable assistance from:
 *                  Alan Cox <alan@lxorguk.ukuu.org.uk>
 *                  Gaylan Ohlhausen <gaylano@worldnet.att.net>
 *                  C. Scott Ananian <cananian@princeton.edu>
 *
 * The information used to write this program was obtained by tracing the
 * interactions of the DOS packet driver and the PE1 using the Linux DOS
 * emulator. I have no idea if the programming model I have derived bears
 * any resemblance to Xircom's. I have not at any stage disassembled the
 * Xircom supplied packet driver.
 *
 *
 * This is ALPHA software -- use at your own risk.
 *
 * This is a preliminary driver, that just checks that what I have worked
 * out about the Xircom PE1 is correct. You should have an Xircom PE1 
 * attached to your Linux machine and powered.
 *
 * Compile it with:
 *
 *      gcc -O -DMODULE -D__KERNEL__ -c pe1.c
 *
 * Load with:
 *
 *      insmod pe1
 *
 * And check your system log for messages! You should find the type of adaptor,
 * the ethernet address and the date/time of manufacture. Please e-mail any
 * success or failures to me, jab@hex.prestel.co.uk .
 *
 *
 */

static char *version = "pe1.c:v0.1 1/4/97 Jonathan Buzzard (jab@hex.prestel.co.uk)\n";


#ifdef MODULE
#include<linux/module.h>
#include<linux/version.h>
#endif

#include<linux/kernel.h>
#include <linux/sched.h>
#include<linux/types.h>
#include<linux/string.h>
#include<linux/interrupt.h>
#include<linux/ioport.h>
#include<asm/io.h>
#include<asm/irq.h>
#include<linux/delay.h>

#include<linux/in.h>
#include<linux/inet.h>
#include<linux/netdevice.h>
#include<linux/etherdevice.h>
#include<linux/skbuff.h>


/* Declare functions */

extern int pe1_probe(struct device *dev);

static struct device pe1_dev = {
	"        ", 
	0, 0, 0, 0,
	0x378, 7,
	0, 0, 0, NULL, pe1_probe
};

static int pe1_base = 0x378;
static int pe1_irq = 7;
static int pe1_irqfound = 0;


/* port i/o support (stollen from parbus) */

#define PE3_BASE(x)     pe3_base

#define r_dtr(x)        inb(PE3_BASE(x))
#define r_str(x)        inb(PE3_BASE(x)+1)
#define r_ctr(x)        inb(PE3_BASE(x)+2)
#define r_epp(x)        inb(PE3_BASE(x)+4)
#define r_fifo(x)       inb(PE3_BASE(x)+0x400)
#define r_ecr(x)        inb(PE3_BASE(x)+0x402)

#define w_dtr(x,y)      outb((y), PE3_BASE(x))
#define w_str(x,y)      outb((y), PE3_BASE(x)+1)
#define w_ctr(x,y)      outb((y), PE3_BASE(x)+2)
#define w_epp(x,y)      outb((y), PE3_BASE(x)+4)
#define w_fifo(x,y)     outb((y), PE3_BASE(x)+0x400)
#define w_ecr(x,y)      outb((y), PE3_BASE(x)+0x402)



/*
 * Special I/O functions for debuggin purposes
 */
void W_dtr(int x, unsigned char y)
{
	outb(y, x);
	printk("write port 0x378 value %02x\n",y);
}
void W_ctr(int x, unsigned char y)
{
	outb(y, x+2);
	printk("write port 0x37a value %02x\n",y);
}
unsigned char R_dtr(int x)
{
	unsigned char temp;

	temp = inb(x);
	printk("read port 0x378 gave %02x\n", temp);
	return temp;
}
unsigned char R_str(int x)
{
	unsigned char temp;

	temp = inb(x+1);
	printk("read port 0x379 gave %02x\n", temp);
	return temp;
}

/*
 * Probe for the presence of a Xircom PE1
 *
 *     returns zero if present, and non-zero if unable to detect
 *     a functioning adaptor.
 */

int pe1_probe(struct device *dev)
{
	int loop,bit;
	unsigned char *current,temp;
	int iosize = (dev->base_addr==0x3bc) ? 3 : 8;

#include "pe1_firm.h"

	/* Check region before the probe */

	if (check_region(dev->base_addr, iosize) < 0) {
		printk("pe1: ERROR ioports 0x%x-0x%x already in use\n",
		        (int) dev->base_addr, (int) dev->base_addr+iosize);
		return -ENODEV;
	}

	/* this is based on trace of what Xircom's PE1 packet driver does. */

	w_dtr(dev->base_addr, 0x00);
	w_ctr(dev->base_addr, 0x00);
	
	/* wait base_addr*2 usecs !!! */
	udelay(dev->base_addr*2);
		
	w_ctr(dev->base_addr, 0x04);
	
	udelay(120000);

	/* upload firmware byte sequence to the adaptor */

	current = pe1_firm;
	for (loop=0;loop<1507;loop++) {
		pe1_firm[loop];
		for (bit=7;bit>=0;bit--) {
			temp = (pe1_firm[loop] == 0) ? 0x04 : 0x06;
			w_ctr(dev->base_addr, temp);
			temp = temp ^ 0x08
			w_ctr(dev->base_addr, temp);
			temp = temp ^ 0x08
			w_ctr(dev->base_addr, temp);
		}
	}

	w_ctr(dev->base_addr, 0x04);
	w_dtr(dev->base_addr, 0x60);
	w_ctr(dev->base_addr, 0x0c);
	w_ctr(dev->base_addr, 0x04);
	w_dtr(dev->base_addr, 0xff);
	w_ctr(dev->base_addr, 0x06);

	r_str(dev->base_addr);
	
	w_ctr(dev->base_addr, 0x04);
	w_dtr(dev->base_addr, 0xff);
	w_dtr(dev->base_addr, 0x80);

	r_str(dev->base_addr);
	w_dtr(dev->base_addr, 0x00);
	r_str(dev->base_addr);

	w_ctr(dev->base_addr, 0x04);
	w_dtr(dev->base_addr, 0xff);


	w_ctr(dev->base_addr, 0x04);
	w_dtr(dev->base_addr, 0x00);
	w_ctr(dev->base_addr, 0x0c);
	w_ctr(dev->base_addr, 0x04);
	w_dtr(dev->base_addr, 0xff);

	w_ctr(dev->base_addr, 0x0c);

	/* for now return failed to stop the module being installed */

	release_region(dev->base_addr, iosize);
	return 1;
}


#ifdef MODULE

int init_module(void)
{
	printk("%s\n", version);
	pe1_dev.base_addr = pe1_base;
	pe1_dev.irq = pe1_irq;
	if (register_netdev(&pe1_dev)!=0)
		return -EIO;
	return 0;
}

void cleanup_module(void)
{
	int iosize = (pe1_dev.base_addr==0x3bc) ? 3 : 8;

	if (MOD_IN_USE)
		printk("%s: device busy, remove delayed\n", pe1_dev.name);
	else {
		unregister_netdev(&pe1_dev);
		release_region(pe1_dev.base_addr, iosize);
	}
	return;
}

#endif
